<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Model;

use Contributte\Translation\Translator;
use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Event\Event;
use Core\Model\Event\EventDispatcher;
use EshopGifts\DI\EshopGiftsExtension;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderDiscount;
use EshopOrders\Model\Entities\OrderGift;
use EshopOrders\Model\Entities\OrderItem;
use EshopOrders\Model\Entities\OrderItemGift;
use EshopOrders\Model\Entities\OrderStatus;
use EshopOrders\Model\Orders;

class DataConverter
{
	public function __construct(
		protected Orders                 $orders,
		protected EntityManagerDecorator $em,
		protected EventDispatcher        $eventDispatcher,
		protected Translator             $translator,
	)
	{
	}

	public function forPrint(array $ids): array
	{
		$statuses = [];
		foreach ($this->em->getRepository(OrderStatus::class)->createQueryBuilder('os')
			         ->select('IDENTITY(os.order) as order, IDENTITY(os.status) as status')
			         ->andWhere('os.order IN (:ids)')
			         ->andWhere('os.deleted = 0 OR os.deleted IS NULL')
			         ->setParameter('ids', $ids)
			         ->orderBy('os.created', 'desc')
			         ->getQuery()->getArrayResult() as $row) {
			$statuses[$row['order']][] = [
				'id' => $row['status'],
			];
		}

		$orders = [];

		$qb = $this->orders->getEr()->createQueryBuilder('o')
			->addSelect('oItems, oItemTexts, iItemsProd, oDel, oInv, os, op, oDisc, oStat')
			->innerJoin('o.orderItems', 'oItems')
			->leftJoin('oItems.product', 'iItemsProd')
			->leftJoin('oItems.orderItemTexts', 'oItemTexts')
			->leftJoin('o.addressDelivery', 'oDel')
			->leftJoin('o.addressInvoice', 'oInv')
			->innerJoin('o.spedition', 'os')
			->innerJoin('o.payment', 'op')
			->leftJoin('o.orderDiscounts', 'oDisc')
			->innerJoin('o.orderStatuses', 'oStat')
			->andWhere('o.id IN (:ids)')
			->setParameter('ids', $ids)
			->orderBy('o.id', 'desc');

		if (class_exists(EshopGiftsExtension::class)) {
			$qb->leftJoin('oItems.gifts', 'oItemsGifts')
				->leftJoin('o.gifts', 'oGifts')
				->addSelect('oItemsGifts, oGifts');
		}

		foreach ($qb->getQuery()->getResult() as $order) {
			/** @var Order $order */
			$row = [
				'id'            => $order->getId(),
				'priceTotal'    => $order->getPrice(),
				'name'          => '',
				'email'         => '',
				'phone'         => '',
				'country'       => '',
				'dic'           => '',
				'created'       => $order->getCreatedTime(),
				'note'          => $order->getMessage(),
				'gifts'         => [],
				'speditionName' => $order->getSpedition() ? $order->getSpedition()->getName() : '',
				'paymentName'   => $order->getPayment() ? $order->getPayment()->getName() : '',
			];

			$addrInv = $order->getAddressInvoice();
			if ($addrInv) {
				$row['name']    = $addrInv->getFirstName() . ' ' . $addrInv->getLastName();
				$row['email']   = $addrInv->getEmail();
				$row['phone']   = $addrInv->getPhone();
				$row['country'] = $addrInv->getCountry() ? mb_strtolower($addrInv->getCountry()->getId()) : '';
				$row['dic']     = $addrInv->getVatNumber();
			}

			foreach ($order->getGifts()->toArray() as $gift) {
				/** @var OrderGift $gift */
				$row['gifts'][] = [
					'code1' => $gift->code1,
					'name'  => $gift->getName(),
				];
			}

			$vats  = [
				21 => [
					'base'  => 0,
					'dph'   => 0,
					'total' => 0,
				],
			];
			$items = [];

			// Produkty
			$orderItems                = $order->getOrderItems()->toArray();
			$event                     = new Event();
			$event->data['orderItems'] = &$orderItems;
			$this->eventDispatcher->dispatch($event, 'eshopOrders.orderItems.loadMoreData');

			foreach ($orderItems as $item) {
				/** @var OrderItem $item */
				$product = $item->getProduct();

				$gifts = [];
				foreach ($item->getGifts()->toArray() as $gift) {
					/** @var OrderItemGift $gift */
					$gifts[] = [
						'code1' => $gift->code1,
						'name'  => $gift->getName(),
					];
				}

				$items[] = [
					'quantity'  => $item->getQuantity(),
					'code1'     => $item->getCode1(),
					'code2'     => $product ? $product->code2 : null,
					'name'      => $item->getOrderItemText()->getName(),
					'code1Star' => $item->getMoreDataValue('stockSupplyAssigned'),
					'note'      => $item->getMoreDataValue('note'),
					'gifts'     => $gifts,
					'hasParent' => (bool) $item->getParent(),
				];
			}

			$priceAfterDiscount = $order->getPriceItems();
			$altSaleName        = $this->translator->translate('eshopAccountant.export.sale');
			foreach ($order->getOrderDiscounts()->toArray() as $discount) {
				/** @var OrderDiscount $discount */
				$amount             = $discount->getPrice();
				$priceAfterDiscount += $amount;
				$items[]            = [
					'quantity'    => 1,
					'code1'       => null,
					'code2'       => null,
					'description' => $discount->getDescription(),
					'name'        => !$discount->getCodeFormatted() ? $altSaleName : ($discount->getCode() . ' - ') . $discount->getName(),
				];
			}

			// Doprava a platba
			$orderSpedition = $order->getSpedition();
			$cmsSpedition   = $orderSpedition->getSpedition();
			$orderPayment   = $order->getPayment();
			$items[]        = [
				'quantity' => 1,
				'code1'    => $cmsSpedition ? $cmsSpedition->getIdent() : null,
				'code2'    => null,
				'name'     => trim($orderSpedition->getName() . ' ' . $orderPayment->getName()),
			];

			if ($statuses[$row['id']][0]['id'] !== OrderStatus::STATUS_CANCELED) {
				foreach ($order->getVatRates() as $rate => $vatRate) {
					$vats[$rate]['base']  = $vatRate['withoutVat'];
					$vats[$rate]['dph']   = $vatRate['total'] - $vatRate['withoutVat'];
					$vats[$rate]['total'] = $vatRate['total'];
				}
			}

			$row['items'] = $items;
			$row['vats']  = $vats;
			$orders[]     = $row;
		}

		$vatTotal = [
			'total' => 0,
			'vats'  => [],
		];

		foreach ($orders as $order) {
			foreach ($order['vats'] as $vatRate => $vat) {
				if (!isset($vatTotal['vats'][$vatRate])) {
					$vatTotal['vats'][$vatRate] = [
						'base'  => 0,
						'dph'   => 0,
						'total' => 0,
					];
				}

				$vatTotal['vats'][$vatRate]['base']  += $vat['base'];
				$vatTotal['vats'][$vatRate]['dph']   += $vat['dph'];
				$vatTotal['vats'][$vatRate]['total'] += $vat['total'];

				$vatTotal['total'] += $vat['total'];
			}
		}

		return [
			'statuses' => $statuses,
			'orders'   => $orders,
			'vatTotal' => $vatTotal,
		];
	}
}
