<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Presenters;

use Core\Model\UI\Form\BaseForm;
use EshopOrders\AdminModule\Components\Customer\CustomerForm;
use EshopOrders\AdminModule\Components\Customer\CustomersGrid;
use EshopOrders\AdminModule\Components\Customer\GroupCustomersForm;
use EshopOrders\AdminModule\Components\Customer\GroupsCustomersGrid;
use EshopOrders\AdminModule\Components\Customer\ICustomerFormFactory;
use EshopOrders\AdminModule\Components\Customer\ICustomersGridFactory;
use EshopOrders\AdminModule\Components\Customer\IGroupCustomersFormFactory;
use EshopOrders\AdminModule\Components\Customer\IGroupsCustomersGridFactory;
use EshopOrders\AdminModule\Components\Customer\Loyalty\ILoyaltyPointsFormFactory;
use EshopOrders\AdminModule\Components\Customer\Loyalty\ILoyaltyPointsGridFactory;
use EshopOrders\AdminModule\Components\Customer\Loyalty\LoyaltyPointsForm;
use EshopOrders\AdminModule\Components\Customer\Loyalty\LoyaltyPointsGrid;
use EshopOrders\AdminModule\Model\Customers;
use EshopOrders\AdminModule\Model\LoyaltyService;
use EshopOrders\Model\Entities\Customer;
use EshopOrders\Model\Entities\GroupCustomers;
use EshopOrders\Model\EshopOrdersConfig;
use EshopOrders\Model\Loyalty\LoyaltyPointsManager;
use Nette\Application\UI\Multiplier;
use Override;

class CustomersPresenter extends BasePresenter
{
	protected ?Customer $customer = null;

	public function __construct(
		protected Customers            $customers,
		protected LoyaltyService       $loyaltyService,
		protected LoyaltyPointsManager $loyaltyPointsManager,
	)
	{
		parent::__construct();
	}

	#[Override]
	protected function startup()
	{
		parent::startup();
		$this['title']->setTitle($this->translator->translate('eshopOrders.menu.customers'));
		$this->setHeader('eshopOrders.menu.customers', 'fa fa-user-friends');
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */

	public function handleRecalculateLoyaltyPoints(int $customerId, string $siteIdent): void
	{
		$this->loyaltyPointsManager->recalculateCurrentPoints($customerId, $siteIdent);

		$this->redirect('this');
	}

	/*******************************************************************************************************************
	 * ============================== Actions
	 */

	public function actionJsonList(): void
	{
		$output = [];

		foreach ($this->customers->getAllArray() as $row) {
			$output[$row['id']] = [
				'id'    => (string) $row['id'],
				'email' => $row['email'],
				'name'  => trim($row['email'] . ' - ' . $row['name'] . ' ' . $row['lastname']),
			];
		}

		$this->sendJson($output);
	}

	public function actionDefault(): void
	{
		$this['navigation']->setData(['header' => [
		]]);

		$this['navigation']->setData(['header' => [
			[
				'title' => 'eshopOrders.menu.addGroupCustomers',
				'link'  => 'Customers:addGroup',
				'ico'   => 'plus',
			],
		]]);
	}

	public function actionEditCustomer(int $id): void
	{
		$customer = $this->getCustomer();

		if (!$customer) {
			$this->error('customer not found');
		}

		$this->template->customer = $customer;

		if (EshopOrdersConfig::load('allowLoyaltyPoints')) {
			$this->template->loyaltyPointsBySite = $this->loyaltyService->getLoyaltyPoints($customer->getId());
		}

		$this->setTitle($this->translator->translate('eshopOrders.title.editCustomer'), $customer->getUser()->getName() . ' ' . $customer->getUser()->getLastname());
		$this->setHeader('eshopOrders.title.editCustomer', 'fa fa-user-friends');
	}

	public function actionAddGroup(): void
	{
		$this->setTitle($this->translator->translate('eshopOrders.title.addGroupCustomers'));
		$this->setHeader('eshopOrders.title.addGroupCustomers', 'fa fa-users');
	}

	public function actionEditGroup(int $id): void
	{
		$group = $this->em->getRepository(GroupCustomers::class)->find($id);

		if (!$group) {
			$this->error('group not found');
		}

		$this->setTitle($this->translator->translate('eshopOrders.title.editGroupCustomers'));
		$this->setHeader('eshopOrders.title.editGroupCustomers', 'fa fa-users');
	}

	protected function getCustomer(): ?Customer
	{
		$customerId = $this->getParameter('id');

		if ($customerId) {
			$this->customer = $this->em->getRepository(Customer::class)->findOneBy(['id' => $customerId]);
		}

		return $this->customer;
	}

	/*******************************************************************************************************************
	 * ============================== Components
	 */

	protected function createComponentCustomersGrid(ICustomersGridFactory $factory): CustomersGrid { return $factory->create(); }

	protected function createComponentGroupsCustomersGrid(IGroupsCustomersGridFactory $factory): GroupsCustomersGrid { return $factory->create(); }

	protected function createComponentCustomerForm(ICustomerFormFactory $factory): CustomerForm
	{
		$control = $factory->create();

		if ($this->getParameter('id')) {
			$control->setCustomer((int) $this->getParameter('id'));
		}

		$control['form']->onSuccessSave[]         = function(BaseForm $form): void {
			$this->redirect('editCustomer', (int) $form->getCustomData('customerId'));
		};
		$control['form']->onSuccessSaveAndClose[] = function(BaseForm $form): void {
			$this->redirect('default');
		};
		$control['form']->onCancel[]              = function(BaseForm $form): void {
			$this->redirect('default');
		};

		return $control;
	}

	protected function createComponentGroupCustomersForm(IGroupCustomersFormFactory $factory): GroupCustomersForm
	{
		$control = $factory->create();

		if ($this->getParameter('id')) {
			$control->setGroupCustomers((int) $this->getParameter('id'));
		}

		$control['form']->onSuccessSave[]         = function(BaseForm $form): void {
			$this->redirect('editGroup', (int) $form->getCustomData('groupCustomersId'));
		};
		$control['form']->onSuccessSaveAndClose[] = function(BaseForm $form): void {
			$this->redirect('default');
		};
		$control['form']->onCancel[]              = function(BaseForm $form): void {
			$this->redirect('default');
		};

		return $control;
	}

	protected function createComponentLoyaltyPointsForm(ILoyaltyPointsFormFactory $factory): LoyaltyPointsForm
	{
		$customer = $this->getCustomer();

		if ($customer) {
			return $factory->create($customer);
		}

		throw new \Exception('customer not found');
	}

	protected function createComponentLoyaltyPointsGrid(ILoyaltyPointsGridFactory $factory): Multiplier
	{
		$customer = $this->getCustomer();
		if (!$this->customer) {
			throw new \Exception('customer not found');
		}

		return new Multiplier(function(string $site) use ($customer, $factory): LoyaltyPointsGrid {
			return $factory->create($customer, $site);
		});
	}
}
