<?php declare(strict_types = 1);

namespace EshopOrders\Console\Loyalty;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Helpers\Arrays;
use Core\Model\Helpers\Caster;
use Doctrine;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderStatus;
use EshopOrders\Model\Loyalty\LoyaltyCalculator;
use EshopOrders\Model\Loyalty\LoyaltyPointsManager;
use Exception;
use Override;
use Symfony\Component\Console\Attribute\AsCommand;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;

#[AsCommand(name: 'eshoporders:loyalty:initialize-points')]
class InitializePoints extends Command
{

	public function __construct(
		public EntityManagerDecorator $em,
		public LoyaltyCalculator      $loyaltyCalculator,
		public LoyaltyPointsManager   $loyaltyPointsManager,
	)
	{
		parent::__construct();
	}

	#[Override]
	protected function configure(): void
	{
		$this->setName('eshoporders:loyalty:initialize-points')
			->addOption('days', null, InputOption::VALUE_REQUIRED)
			->addOption('site', null, InputOption::VALUE_REQUIRED)
			->setDescription('Loyalty points - initialize points for exist data');
	}

	#[Override]
	protected function execute(
		InputInterface  $input,
		OutputInterface $output,
	): int
	{
		$output->writeln('START INITIALIZE POINTS');

		try {
			$days      = Caster::toInt($input->getOption('days'));
			$siteIdent = Caster::toString($input->getOption('site'));

			$connection = $this->em->getConnection();
			$dateFrom   = (new \DateTime())->modify('-' . $days . ' days');

			$existsCustomersData = $connection->fetchFirstColumn("SELECT customer_id 
					FROM eshop_orders__customer_loyalty 
					WHERE site_id = :siteIdent", [
				'siteIdent' => $siteIdent,
			]);
			$cancelledOrders     = $connection->fetchFirstColumn("SELECT order_id 
					FROM eshop_orders__order_status 
					WHERE created >= :dateFrom AND status_id = :status", [
				'dateFrom' => $dateFrom->format('Y-m-d 00:00:00'),
				'status'   => OrderStatus::STATUS_CANCELED,
			]);

			$pointsToSet = [];

			foreach ($connection->iterateAssociative("SELECT o.id, o.customer_id
					FROM eshop_orders__order o
					INNER JOIN eshop_orders__order_status os ON os.order_id = o.id AND os.status_id = :status AND os.created >= :dateFrom
					WHERE o.site_id = :siteIdent AND o.customer_id IS NOT NULL", [
				'status'    => OrderStatus::STATUS_CREATED,
				'dateFrom'  => $dateFrom->format('Y-m-d 00:00:00'),
				'siteIdent' => $siteIdent,
			]) as $row) {
				/** @var array $row */
				$orderId    = $row['id'];
				$customerId = $row['customer_id'];

				if (Arrays::contains($cancelledOrders, $orderId) || Arrays::contains($existsCustomersData, $customerId)) {
					continue;
				}

				/** @var Order $order */
				$order = $this->em->getRepository(Order::class)->find($orderId);

				$points = $this->loyaltyCalculator->calculate($order->getPriceItemsDiscount(), $siteIdent);
				if ($points > 0) {
					if (!isset($pointsToSet[$customerId])) {
						$pointsToSet[$customerId] = 0;
					}

					$pointsToSet[$customerId] += $points;
				}
			}

			foreach ($pointsToSet as $customerId => $points) {
				$this->loyaltyPointsManager->modify($points, $customerId, $siteIdent, 'admin');
			}

			$output->writeLn('<info>[OK] - points initialized</info>');
		} catch (Exception $e) {
			$output->writeln('<error>[ERROR] - ' . $e->getMessage() . '</error>');
		}

		return 0;
	}
}
