<?php declare(strict_types = 1);

namespace EshopOrders\CronModule\Model;

use Core\Model\Entities\EntityManagerDecorator;

class AcquisitionRetention
{
	public function __construct(
		protected EntityManagerDecorator $em,
	)
	{
	}

	public function findAcquisitionProducts(string $siteIdent, string $from, string $to): array
	{
		$conn = $this->em->getConnection();

		$emailsOrder = [];
		$orders      = [];

		foreach ($conn->executeQuery("SELECT o.id, oai.email as invoice_email, oad.email as delivery_email FROM eshop_orders__order o
			INNER JOIN eshop_orders__order_status os ON os.order_id = o.id AND os.status_id = 'created' AND os.created >= ? AND os.created <= ?
			LEFT JOIN eshop_orders__order_address oai ON oai.id = o.address_invoice_id
			LEFT JOIN eshop_orders__order_address oad ON oad.id = o.address_delivery_id
			WHERE o.site_id = ?", [$from . ' 00:00:00', $to . ' 23:59:59', $siteIdent])->iterateAssociative() as $row) {
			/** @var array $row */
			$email = $row['invoice_email'] ?: $row['delivery_email'];

			if (isset($emailsOrder[$email])) {
				continue;
			}

			$exist = $conn->executeQuery("SELECT o.id FROM eshop_orders__order o
    			LEFT JOIN eshop_orders__order_address inv ON inv.id = o.address_invoice_id
    			LEFT JOIN eshop_orders__order_address del ON del.id = o.address_delivery_id
				WHERE (inv.email = ? OR del.email = ?) AND o.id != ?
				LIMIT 1", [$email, $email, $row['id']])->fetchOne();

			$emailsOrder[] = $row['id'];

			if (!$exist) {
				$orders[] = $row['id'];
			}
		}

		if (empty($orders)) {
			return [];
		}

		return $this->findProductsInOrders($orders);
	}

	public function findRetentionProducts(string $siteIdent, string $from, string $to): array
	{
		$conn = $this->em->getConnection();

		$isProcessed = [];
		$orders      = [];

		foreach ($conn->executeQuery("SELECT o.id, oai.email as invoice_email, oad.email as delivery_email FROM eshop_orders__order o
			INNER JOIN eshop_orders__order_status os ON os.order_id = o.id AND os.status_id = 'created' AND os.created >= ? AND os.created <= ?
			LEFT JOIN eshop_orders__order_address oai ON oai.id = o.address_invoice_id
			LEFT JOIN eshop_orders__order_address oad ON oad.id = o.address_delivery_id
			WHERE o.site_id = ?", [$from . ' 00:00:00', $to . ' 23:59:59', $siteIdent])->iterateAssociative() as $row) {
			/** @var array $row */
			$email = $row['invoice_email'] ?: $row['delivery_email'];

			if (isset($isProcessed[$email])) {
				continue;
			}

			$isProcessed[$email] = true;

			$count = $conn->executeQuery("SELECT COUNT(*) as `count` FROM eshop_orders__order o
    			LEFT JOIN eshop_orders__order_address inv ON inv.id = o.address_invoice_id
    			LEFT JOIN eshop_orders__order_address del ON del.id = o.address_delivery_id
				WHERE (inv.email = ? OR del.email = ?)", [$email, $email])->fetchOne();

			if ($count > 1) {
				$orders[] = $row['id'];
			}
		}

		if (empty($orders)) {
			return [];
		}

		return $this->findProductsInOrders($orders);
	}

	protected function findProductsInOrders(array $orders): array
	{
		$products = [];
		$conn     = $this->em->getConnection();

		foreach (array_chunk($orders, 200) as $chunk) {
			foreach ($conn->executeQuery("SELECT oi.product_id, oi.code1, oi.quantity, oi.vat_rate, oi.price 
					FROM eshop_orders__order_item oi
					WHERE oi.order_id IN (" . implode(',', $chunk) . ")")->iterateAssociative() as $row) {
				/** @var array $row */
				$productId     = $row['product_id'];
				$quantity      = (int) $row['quantity'];
				$pricePerPiece = (float) $row['price'];
				$price         = round($quantity * $pricePerPiece, 2);

				if (!isset($products[$productId])) {
					$products[$row['product_id']] = [
						'code1'    => (string) $row['code1'],
						'quantity' => $quantity,
						'vatRate'  => (int) $row['vat_rate'],
						'price'    => $price,
					];
				} else {
					$products[$productId]['quantity'] += $quantity;
					$products[$productId]['price']    += $price;
				}
			}
		}

		uasort($products, static function($a, $b) {
			return $b['price'] <=> $a['price'];
		});

		if (!empty($products)) {
			$productsIds = array_keys($products);

			foreach (array_chunk($productsIds, 200) as $chunk) {
				foreach ($conn->executeQuery("SELECT pt.id, pt.name FROM eshop_catalog__product_texts pt
					WHERE pt.id IN (" . implode(',', $chunk) . ") AND pt.lang = ?", ['cs'])->iterateAssociative() as $row) {
					/** @var array $row */
					$products[$row['id']]['name'] = $row['name'];
				}
			}
		}

		return $products;
	}
}
