<?php declare(strict_types = 1);

namespace EshopOrders\CronModule\Presenters;

use Core\Model\Entities\EntityManagerDecorator;
use EshopOrders\CronModule\Model\AcquisitionRetention;
use EshopOrders\CronModule\Model\Orders as CronOrders;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\EshopOrdersConfig;
use EshopOrders\Model\Orders;
use Nette\Security\Passwords;

class DefaultPresenter extends BasePresenter
{
	public function __construct(
		protected Orders                 $ordersService,
		protected EntityManagerDecorator $em,
		protected AcquisitionRetention   $acquisitionRetention,
		protected CronOrders             $cronOrders,
	)
	{
		parent::__construct();
	}

	public function actionSendSurvey(): void
	{
		ini_set('memory_limit', '4g');
		$config    = EshopOrdersConfig::loadInt('sendSurveyAfterOrder') ?: null;
		$bulkLimit = EshopOrdersConfig::loadInt('surveyBulkLimit') ?: 20;

		if ($config === null) {
			$this->terminate();
		}

		$qb = $this->ordersService->getOrdersWithoutSurveySent($config, (int) $bulkLimit);

		$i = 1;
		foreach ($qb->getQuery()->toIterable() as $order) {
			/** @var Order $order */
			$this->ordersService->sendSurvey($order);
			$i++;

			if ($i % 20 === 0) {
				$this->em->clear();
				gc_collect_cycles();
			}
		}

		$this->terminate();
	}

	public function actionAcquisition(
		string $key,
		string $siteIdent,
		string $from,
		string $to
	): void
	{
		set_time_limit(300);

		$password  = new Passwords();
		$keyVerify = $password->verify(EshopOrdersConfig::loadString('cronUrlHash') . $siteIdent, $key);

		if (!$keyVerify) {
			$this->error();
		}

		header('Content-Encoding: UTF-8');
		header("content-type:application/csv;charset=UTF-8");
		header("Content-Disposition:attachment;filename=\"acquisition-products-{$from}-{$to}.csv\"");
		header('Content-Transfer-Encoding: binary');

		$fp = fopen('php://output', 'wb');

		if (!$fp) {
			$this->error();
		}

		fputs($fp, "\xEF\xBB\xBF");

		foreach ($this->acquisitionRetention->findAcquisitionProducts($siteIdent, $from, $to) as $id => $row) {
			fputcsv($fp, [
				$id,
				$row['name'],
				$row['code1'],
				$row['quantity'],
				$row['price'],
				$row['vatRate'],
			], ';');
		}

		fclose($fp);

		exit;
	}

	public function actionRetention(
		string $key,
		string $siteIdent,
		string $from,
		string $to
	): void
	{
		set_time_limit(300);

		$password  = new Passwords();
		$keyVerify = $password->verify(EshopOrdersConfig::loadString('cronUrlHash') . $siteIdent, $key);

		if (!$keyVerify) {
			$this->error();
		}

		header('Content-Encoding: UTF-8');
		header("content-type:application/csv;charset=UTF-8");
		header("Content-Disposition:attachment;filename=\"retention-products-{$from}-{$to}.csv\"");
		header('Content-Transfer-Encoding: binary');

		$fp = fopen('php://output', 'wb');

		if (!$fp) {
			$this->error();
		}

		fputs($fp, "\xEF\xBB\xBF");

		foreach ($this->acquisitionRetention->findRetentionProducts($siteIdent, $from, $to) as $id => $row) {
			fputcsv($fp, [
				$id,
				$row['name'],
				$row['code1'],
				$row['quantity'],
				$row['price'],
				$row['vatRate'],
			], ';');
		}

		fclose($fp);

		exit;
	}

	public function actionOrdersByDate(
		string  $key,
		string  $siteIdent,
		string  $from,
		string  $to,
		?string $lang = null
	): void
	{
		set_time_limit(300);

		$password  = new Passwords();
		$keyVerify = $password->verify(EshopOrdersConfig::loadString('cronUrlHash') . $siteIdent, $key);

		if (!$keyVerify) {
			$this->error();
		}

		$fileName = "orders-by-date-{$from}-{$to}";

		if ($lang) {
			$fileName .= "-{$lang}";
		}

		header('Content-Encoding: UTF-8');
		header("content-type:application/csv;charset=UTF-8");
		header("Content-Disposition:attachment;filename=\"{$fileName}.csv\"");
		header('Content-Transfer-Encoding: binary');

		$fp = fopen('php://output', 'wb');

		if (!$fp) {
			$this->error();
		}

		fputs($fp, "\xEF\xBB\xBF");

		$i = 0;
		foreach ($this->cronOrders->ordersByDate($siteIdent, $from, $to, $lang) as $row) {
			if ($i === 0) {
				fputcsv($fp, array_keys($row), ';');
			}

			fputcsv($fp, array_values($row), ';');

			$i++;
		}

		exit;
	}
}
