<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Components\Cart;

use Core\Model\Event\ComponentTemplateEvent;
use Core\Model\UI\BaseControl;
use Currency\Model\Currencies;
use EshopCatalog\Model\Config;
use EshopOrders\FrontModule\Model\CartFacade;
use EshopOrders\FrontModule\Model\Carts;
use EshopOrders\FrontModule\Model\Event\UpdatedCartItemEvent;
use EshopOrders\Model\EshopOrdersConfig;
use Nette\Application\UI\Multiplier;
use Nette\ComponentModel\IComponent;
use Nette\DI\Container;
use Nette\Http\Request;
use Override;

class CartDetail extends BaseControl
{
	public bool $showCartSteps = true;

	public function __construct(
		protected CartFacade             $cartFacade,
		protected ICartDetailItemFactory $cartDetailItemFactory,
		protected Container              $container,
		protected Request                $request,
		protected Currencies             $currencies
	)
	{
		// add discount form control to this component
		if ($this->container->hasService('eshopSales.front.discountsGridFactory')) {
			$discountsGridControl = $this->container->getService('eshopSales.front.discountsGridFactory');
			$this->addComponent($discountsGridControl->create(), 'discountsGrid');
		}

		if ($this->container->hasService('eshopSales.front.discountFormFactory')) {
			$discountFormControl = $this->container->getService('eshopSales.front.discountFormFactory');
			$this->addComponent($discountFormControl->create(), 'discountForm');
		}
	}

	#[Override]
	protected function attached(IComponent $presenter): void
	{
		parent::attached($presenter);

		$this->eventDispatcher->addListener('eshopOrders.cartUpdateItem', function(): void {
			$cart = $this->cartFacade->getCart();

			if ($cart->getItemsCount() !== 0) {
				$this->redrawControl('wrap');
			} else {
				$this->getPresenter()->redrawControl('cart');
			}
		});

		$this->eventDispatcher->addListener('eshopSales.discountCodesChanged', function(): void {
			$this->cartFacade->getCartsService()->reloadCart();
			$parent = $this->getParent();

			if ($parent && method_exists($parent, 'redrawControl')) {
				$parent->redrawControl('cart');
			}
		});
	}

	public function handleRemove(int $itemId): void
	{
		$cartItems = $this->cartFacade->getCartsService()->getCurrentCart()->getCartItems();
		$item      = $cartItems[$itemId] ?? null;

		if ($item) {
			$event = new UpdatedCartItemEvent(
				(string) $item->getId(),
				0,
				(int) $item->quantity,
				[],
				$item->getProductId(),
			);

			$this->eventDispatcher->dispatch($event, 'eshopOrders.cartUpdateItem');
		}
	}

	public function handleChooseOrderGift(int $giftId): void
	{
		$this->cartFacade->getCartsService()->setOrderGift($giftId);
		$this->redrawControl('giftsList');
	}

	public function handleRemoveOrderGift(): void
	{
		$this->cartFacade->getCartsService()->setOrderGift();
		$this->redrawControl('giftsList');
	}

	public function handleVerifyAgeConfirm(): void
	{
		$this->cartFacade->getCartsService()->getSession()->set(
			Carts::sessionVerifyAgeModelShowed,
			true,
			'1 hour',
		);
	}

	public function handleVerifyAgeRemoveProducts(): void
	{
		$cartItems = $this->cartFacade->getCart()->getCartItems();
		foreach ($cartItems as $item) {
			if ($item->getProduct() && $item->getProduct()->verifyAge) {
				$this->handleRemove($item->getId());
			}
		}

		$this->redrawControl('wrap');
	}

	public function render(): void
	{
		$cart                          = $this->cartFacade->getCart();
		$this->template->cart          = $cart;
		$this->template->orderGift     = $cart->getOrderGift();
		$this->template->showCartSteps = $this->showCartSteps;

		$style = EshopOrdersConfig::load('cartDetailFreeSped.style', 1);
		if ($style === 1) {
			$freeSped                               = $this->cartFacade->countFreeSpedition();
			$this->template->freeSpeditionPrice     = $freeSped['minPrice'];
			$this->template->freeSpeditionRemaining = $freeSped['remaining'];
		} else if ($style === 2) {
			$this->template->freeSped = $this->cartFacade->countFreeSpeditionMilestones();
		}

		if (Config::load('product.allowVerifyAge')) {
			$cartsService     = $this->cartFacade->getCartsService();
			$sessionVerifyAge = $cartsService->getSession()->get(Carts::sessionVerifyAgeModelShowed);

			if ($sessionVerifyAge !== true) {
				foreach ($cart->getCartItems() as $item) {
					if ($item->verifyAge) {
						$this->template->popupVerifyAge = true;
						break;
					}
				}
			}
		}

		$this->eventDispatcher->dispatch(new ComponentTemplateEvent($this->template, $this), 'eshopOrders.cartDetail.render');

		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ======================== Components
	 */

	protected function createComponentCartDetailItem(): Multiplier
	{
		return new Multiplier(function($item) {
			$control = $this->cartDetailItemFactory->create();
			$control->setItem((int) $item);

			return $control;
		});
	}
}

