<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Components\Customer;

use Core\Model\Event\CreateFormEvent;
use Core\Model\Event\FormSuccessEvent;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopOrders\FrontModule\Model\Customers;
use EshopOrders\Model\Entities\Customer;
use EshopOrders\Model\EshopOrdersConfig;
use Nette\Forms\Form;
use Nette\Utils\ArrayHash;
use Users\Model\Entities\User;
use Users\Model\UsersConfig;

class AccountSettings extends BaseControl
{
	protected Customer $customer;

	public function __construct(
		public User         $user,
		protected Customers $customers,
	)
	{
		$this->customer = $this->customers->getOrCreateCustomer($user->getId(), $user->getEmail(), $user->getName(), $user->getLastname());
	}

	public function render(): void
	{
		$this->template->customer = $this->customer;
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax();

		$form->addText('firstName', 'eshopOrdersFront.accountSettings.firstName');
		$form->addText('lastName', 'eshopOrdersFront.accountSettings.lastName');
		$form->addPhone('phone', 'eshopOrdersFront.accountSettings.phone');
		$form->addEmail('email', 'eshopOrdersFront.accountSettings.email');

		if (!UsersConfig::load('user.hideBirthday')) {
			$form->addText('birthday', 'eshopOrdersFront.accountSettings.birthday')
				->setHtmlType('date');
		}

		$password = $form->addPassword('password', 'eshopOrdersFront.accountSettings.password');
		$form->addPassword('passwordConfirm', 'eshopOrdersFront.accountSettings.passwordConfirm')
			->addRule(Form::EQUAL, 'eshopOrdersFront.accountSettings.passwordConfirmFail', $password);

		if (UsersConfig::load('showNewsletterSubscriber') || UsersConfig::load('registerForm.showNewsletterSubscriber')) {
			$form->addCheckbox('newsletterSubscribed', 'eshopOrdersFront.accountSettings.newsletterSubscribed');
		}

		$event                 = new CreateFormEvent($form, $this->getPresenterIfExists() ? $this->template : null);
		$event->data['entity'] = $this->user;
		$this->eventDispatcher->dispatch($event, self::class . '::createForm');

		$form->addSubmit('submit', 'default.save')
			->setBtnClass('btn-success primary-bg-color');

		$d = [
			'firstName'            => $this->user->name,
			'lastName'             => $this->user->lastname,
			'email'                => $this->user->email,
			'phone'                => $this->user->phone,
			'birthday'             => !UsersConfig::load('user.hideBirthday') && $this->user->birthday ? $this->user->birthday->format('Y-m-d') : null,
			'newsletterSubscribed' => $this->user->newsletterSubscribed,
		];

		$form->setDefaults($d);

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		try {
			$user     = $this->user;
			$customer = $this->customer;

			$user->name                 = $values->firstName;
			$user->lastname             = $values->lastName;
			$user->email                = $values->email;
			$user->phone                = $values->phone;
			$user->newsletterSubscribed = (int) $values->newsletterSubscribed;

			if (!UsersConfig::load('user.hideBirthday')) {
				$birthday       = $values->birthday ? \DateTime::createFromFormat('Y-m-d', $values->birthday) : null;
				$user->birthday = $birthday ?: null;
			}

			$customer->setPhone($values->phone);

			if ($values->password) {
				$user->setPassword($values->password);
			}

			$event                           = new FormSuccessEvent($form, $values, $this->template, $this->presenter);
			$event->custom['entity']         = $user;
			$event->custom['entityCustomer'] = $customer;
			$this->eventDispatcher->dispatch($event, self::class . '::formSuccess');

			$this->em->persist($user);
			$this->em->persist($customer);

			$this->em->flush();
			$this->presenter->flashMessageSuccess('eshopOrdersFront.accountSettings.updated');
		} catch (\Exception $e) {
			$form->addError($e->getMessage());
			$this->redrawControl('form');

			return false;
		}

		$this->redirect('this');
	}

	public function getAddressesOptions(): array
	{
		$addresses = [
			'delivery' => '',
			'invoice'  => '',
		];

		$customer = $this->customer;

		$deliveryDefault = $customer->getAddressDelivery() ? $this->customer->getAddressDelivery()->getId() : null;
		$invoiceDefault  = $customer->getAddressInvoice() ? $this->customer->getAddressInvoice()->getId() : null;

		foreach ($this->customer->addressesAvailable as $addr) {
			if (EshopOrdersConfig::load('customer.disableModifyInvoiceAddress')
				&& $invoiceDefault === $addr->getId()) {
				continue;
			}

			$str                   = $addr->getDetailString();
			$addresses['delivery'] .= '<option value="' . $addr->getId() . '"' . ($deliveryDefault === $addr->getId() ? ' selected' : '') . '>' . $addr->getDetailString() . '</option>';
			$addresses['invoice']  .= '<option value="' . $addr->getId() . '"' . ($invoiceDefault === $addr->getId() ? ' selected' : '') . '>' . $addr->getDetailString() . '</option>';
		}

		return $addresses;
	}
}

