<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Components\Customer;

use Core\Model\Event\Event;
use Core\Model\UI\BaseControl;
use EshopOrders\FrontModule\Model\Customers;
use EshopOrders\FrontModule\Model\OrderFacade;
use EshopOrders\Model\Entities\Customer;
use EshopOrders\Model\EshopOrdersConfig;
use EshopOrders\Model\Orders;
use Nette\Application\Attributes\Persistent;
use Nette\Utils\Paginator;

class OrdersList extends BaseControl
{
	#[Persistent]
	public int $page = 1;

	public int $limit = 8;

	protected ?Customer  $customer  = null;
	protected ?Paginator $paginator = null;

	public function __construct(
		protected Orders      $ordersService,
		protected Customers   $customersService,
		protected OrderFacade $orderFacade,
	)
	{
	}

	public function render(): void
	{
		$this->template->setFile($this->getTemplateFile());

		$customer = $this->getCustomer();

		if (!$customer instanceof Customer) {
			return;
		}

		$paginator = $this->getPaginator();
		$orders    = $this->ordersService->getByCustomer($customer, $paginator->getItemsPerPage(), $paginator->getOffset());

		$this->template->orders    = $orders;
		$this->template->paginator = $paginator;
		$this->template->showUser  = EshopOrdersConfig::load('showOrdersOfChildrenUsers');

		$this->template->render($this->getTemplateFile());
	}

	public function handlePaginator(int $page): void
	{
		$this->page = $page;

		$this->redrawControl('list');
		$this->redrawControl('paginator');
	}

	public function getPaginator(): Paginator
	{
		if (!$this->paginator instanceof Paginator) {
			$itemsPerPage = $this->limit;

			$paginator = new Paginator();
			$paginator->setItemCount($this->getCustomer() instanceof Customer ? $this->ordersService->getOrdersCountByCustomer($this->getCustomer()) : 0);
			$paginator->setItemsPerPage($itemsPerPage);
			$paginator->setPage($this->page);

			$this->paginator = $paginator;
		}

		return $this->paginator;
	}

	protected function getCustomer(): ?Customer
	{
		if (!$this->customer instanceof Customer) {
			$userId         = (int) $this->getPresenter()->getUser()->id;
			$this->customer = $this->customersService->getByUserId($userId);
		}

		return $this->customer;
	}
}

