<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Model\Dao;

use Core\Model\Helpers\Caster;
use Core\Model\Parameters;
use EshopCatalog\Model\Config;
use EshopCatalog\Model\Config as EshopCatalogConfig;
use EshopCatalog\Model\Entities\Availability;
use EshopOrders\Model\PriceCalculator\PriceCalculator;
use EshopOrders\Model\PriceCalculator\PriceCalculatorDiscount;
use EshopOrders\Model\PriceCalculator\PriceCalculatorItem;
use Nette\Utils\Arrays;
use Nette\Utils\Floats;

class Cart
{
	public ?int       $id                   = null;
	public ?string    $ident                = null;
	public ?array     $cartItems            = null;
	public ?Spedition $spedition            = null;
	public ?Payment   $payment              = null;
	public array      $discounts            = [];
	public array      $gifts                = [];
	public ?int       $orderGiftId          = null;
	public float      $minimalOrderPrice    = 0;
	protected array   $cCalculatedDiscounts = [];
	public ?string    $name                 = null;
	public string     $type                 = 'default';

	protected ?bool $cDisableDeliveryBoxes = null;

	/** @var PaymentSpedition[] */
	public array $paymentSpeditionFreeCombinations = [];

	public PriceCalculator $priceCalculator;

	public function __construct(string $currency)
	{
		$this->priceCalculator = new PriceCalculator($currency);
	}

	public function setId(int $id): self
	{
		$this->id = $id;

		return $this;
	}

	public function setIdent(?string $ident): self
	{
		$this->ident = $ident;

		return $this;
	}

	/** @param CartItem[] $cartItems */
	public function setCartItems(array $cartItems): self
	{
		$this->cDisableDeliveryBoxes = null;

		$this->cartItems = $cartItems;

		foreach ($this->cartItems as $k => $cartItem) {
			$item = new PriceCalculatorItem(
				$cartItem->title,
				$cartItem->price,
				$cartItem->getQuantity(),
				$cartItem->getVatRate(),
				$this->priceCalculator->getCurrency(),
				$cartItem->discountDisabled
			);

			foreach ($cartItem->getChilds() as $child) {
				$childItem = new PriceCalculatorItem(
					$child->title,
					$child->price,
					(int) $child->getData('baseQuantity') ?: $child->getQuantity(),
					$child->getVatRate(),
					$this->priceCalculator->getCurrency(),
					$child->discountDisabled
				);

				$item->addChild((string) $child->ident, $childItem);

				$childItem = clone $childItem;
				$childItem->setQuantity($child->getQuantity());
				$child->priceCalculatorItem = $childItem;

				//				$childItem = clone $cartItem;
				//				$childItem->setQuantity($child->getQuantity());
				//				$this->priceCalculator->addItem((string) $child->ident, $childItem);
			}

			$cartItem->priceCalculatorItem = $item;

			$this->priceCalculator->addItem((string) $k, $item);
		}

		return $this;
	}

	public function getId(): int
	{
		return $this->id;
	}

	public function getIdent(): ?string
	{
		return $this->ident;
	}

	/** @return CartItem[] */
	public function getCartItems(): array
	{
		return $this->cartItems ?? [];
	}

	public function changeItemQuantity(string $key, int $quantity): void
	{
		$this->cartItems[$key]->quantity = $quantity;
		$this->priceCalculator->changeItemQuantity($key, $quantity);
	}

	public function removeItemChild(string $itemKey, string $childKey): void
	{
		$this->cartItems[$itemKey]->removeChild($childKey);
		$this->priceCalculator->removeItem($childKey);
	}

	public function getTotalQuantity(): int
	{
		$result = 0;

		foreach ($this->getCartItems() as $item) {
			$result += $item->getQuantity();
		}

		return $result;
	}

	public function getCartItemByProductId(int $id): ?CartItem
	{
		foreach ($this->getCartItems() as $item) {
			if ($item->productId === $id) {
				return $item;
			}
		}

		return null;
	}

	public function getCartItemByIdent(string $ident): ?CartItem
	{
		foreach ($this->getCartItems() as $item) {
			if ($item->ident === $ident) {
				return $item;
			}
		}

		return null;
	}

	/** @return Gift[] */
	public function getGifts(): array { return $this->gifts; }

	public function addGift(Gift $gift): self
	{
		$this->gifts[$gift->id] = $gift;

		return $this;
	}

	/**
	 * Vrátí pouze cenu položek košíku (bez slev, dopravy a platby)
	 */
	public function getCartItemsPrice(bool $withVat = true): float
	{
		$priceTotal = 0;
		foreach ($this->cartItems as $item) {
			$priceTotal += $withVat ? $item->getTotalPrice() : $item->getTotalPriceWithoutVat();
		}

		return $priceTotal;
	}

	/**
	 * @return CartItem[]
	 */
	public function getCartItemsForGifts(): array
	{
		$items = [];
		foreach ($this->cartItems as $item) {
			if ($item->product && !$item->product->orderGiftsAllowed) {
				continue;
			}

			$items[] = $item;
		}

		return $items;
	}

	public function getPriceForFreeSpedition(): float
	{
		$priceTotal = 0;
		foreach ($this->cartItems as $item) {
			if (EshopCatalogConfig::load('product.allowModifySpedition', false) && $item->disableCalculateFreeSpedition) {
				continue;
			}
			$priceTotal += $item->getTotalPrice();
		}

		foreach ($this->priceCalculator->discountCodes as $discount) {
			$priceTotal += $discount->getValue()->getAmount()->toFloat();
		}

		return $priceTotal;
	}

	/**
	 * Vrátí pouze cenu položek po slevě (bez dopravy a platby)
	 */
	public function getPriceItems(): float
	{
		return $this->priceCalculator->getWithVatItems()->getAmount()->toFloat();
	}

	/**
	 * Vrátí celkovou cenu s DPH (polozky, doprava, platba)
	 */
	public function getPriceTotalWithoutDiscounts(): float
	{
		return $this->priceCalculator->getWithVatTotalWithoutDiscounts()->getAmount()->toFloat();
	}

	/**
	 * Vrátí kompletní cenu košíku po slevě s dopravou a platbou
	 */
	public function getPriceTotal(): float
	{
		return $this->priceCalculator->getWithVatTotal()->getAmount()->toFloat();
	}

	public function getPriceItemsWithoutVat(): float
	{
		return $this->priceCalculator->getWithoutVatItems()->getAmount()->toFloat();
	}

	public function getPriceTotalWithoutVat(): float
	{
		return $this->priceCalculator->getWithoutVatTotal()->getAmount()->toFloat();
	}

	/**
	 * Vrátí kompletní cenu košíku bez slev s dopravou a platbou
	 */
	public function getPriceTotalWithoutVatWithoutDiscounts(): float
	{
		return $this->priceCalculator->getWithoutVatTotalWithoutDiscounts()->getAmount()->toFloat();
	}

	public function getItemsCount(): int
	{
		return $this->priceCalculator->getItemsCount();
	}

	public function addDiscount(string $key, Discount $discount): self
	{
		$this->discounts[$key] = $discount;

		$calculatorItem = new PriceCalculatorDiscount(
			$discount->getType(),
			$discount->discount,
			$this->priceCalculator->getCurrency()
		);

		$this->priceCalculator->addDiscountCode((string) $key, $calculatorItem);

		return $this;
	}

	public function removeDiscount(string $key): self
	{
		unset($this->discounts[$key]);

		return $this;
	}

	public function getDiscount(string $key): ?Discount { return $this->discounts[$key] ?? null; }

	/** @return Discount[] */
	public function getDiscounts(): iterable
	{
		// Přepočítá slevu, může tam být uložena sleva bez DPH
		$this->calculateDiscounts();

		$discounts = [];
		foreach ($this->discounts as $k => $discount) {
			$discounts[$k] = $discount;
		}

		return $discounts;
	}

	public function setSpedition(Spedition $spedition): self
	{
		$spedition->setCart($this);
		$this->spedition = $spedition;

		$calculatorItem = new PriceCalculatorItem(
			$spedition->getName(),
			$spedition->getPriceActual($this),
			1,
			$spedition->vat,
			$this->priceCalculator->getCurrency(),
			false
		);
		$this->priceCalculator->setSpedition($calculatorItem);

		return $this;
	}

	public function setPayment(Payment $payment): self
	{
		$payment->setCart($this);
		$this->payment = $payment;

		$calculatorItem = new PriceCalculatorItem(
			$payment->getName(),
			$payment->getPriceActual($this),
			1,
			$payment->vat,
			$this->priceCalculator->getCurrency(),
			false
		);
		$this->priceCalculator->setPayment($calculatorItem);

		return $this;
	}

	public function calculateDiscounts(bool $withVat = true): float
	{
		$k = (string) $withVat;

		if (!isset($this->cCalculatedDiscounts[$k])) {
			$price = 0;
			foreach ($this->discounts as $discount) {
				$price += $discount->calculateDiscount();
			}

			$this->cCalculatedDiscounts[$k] = $price;
		}

		return $this->cCalculatedDiscounts[$k];
	}

	public function calculateDiscount(Discount $discount, bool $withVat = true): float
	{
		return $discount->calculateDiscount();
	}

	/** @return CartItem[] */
	public function getProductsForDiscount(): array
	{
		$items = [];

		foreach ($this->cartItems as $k => $item) {
			if ($item->discountDisabled) {
				continue;
			}

			$items[$k] = $item;
		}

		return $items;
	}

	public function hasOversizedProduct(): bool
	{
		foreach ($this->cartItems as $item) {
			if (
				($item->product && $item->product->isOversize === true)
				|| $item->getData('isOversize') === true
			) {
				return true;
			}
		}

		return false;
	}

	public function hasDisabledPickUpSpedition(): bool
	{
		foreach ($this->cartItems as $item) {
			if ($item->product && $item->product->hasDisabledPickupPoints()) {
				return true;
			}
		}

		return false;
	}

	public function hasDisabledDeliveryBoxes(): bool
	{
		if ($this->cDisableDeliveryBoxes === null) {
			$this->cDisableDeliveryBoxes = false;
			if (Config::load('product.allowModifySpedition')) {
				foreach ($this->getCartItems() as $item) {
					if ($item->getProduct() && $item->getProduct()->disableDeliveryBoxes) {
						$this->cDisableDeliveryBoxes = true;
						break;
					}
				}
			}
		}

		return $this->cDisableDeliveryBoxes;
	}

	public function getOrderGift(): ?Gift
	{
		if (!$this->orderGiftId || !array_key_exists($this->orderGiftId, $this->gifts)) {
			return null;
		}

		$gift = $this->gifts[$this->orderGiftId];

		if ($this->isEnabledGiftMilestones()) {
			$cartItems  = $this->getCartItemsForGifts();
			$itemsPrice = 0;
			foreach ($cartItems as $item) {
				$itemsPrice += $item->getTotalPrice();
			}

			return Floats::isGreaterThanOrEqualTo($itemsPrice, $gift->priceFrom) ? $gift : null;
		}

		return $gift;
	}

	public function getRemainingValueToCompleteOrder(): float
	{
		return $this->minimalOrderPrice > 0 ? $this->minimalOrderPrice - $this->getCartItemsPrice() : 0;
	}

	public function hasPreorderProduct(): bool
	{
		foreach ($this->getCartItems() as $item) {
			if ($item->getProduct() && $item->getProduct()->getAvailability()->getIdent() === Availability::PREORDER) {
				return true;
			}
		}

		return false;
	}

	public function getTotalWeight(): int
	{
		$weight = 0;

		foreach ($this->getCartItems() as $item) {
			$weight += $item->getTotalWeight();
		}

		return $weight;
	}

	public function getTotalWeightInKG(): float
	{
		$totalWeight = $this->getTotalWeight();

		if ($totalWeight > 0) {
			return round($totalWeight / 1000, 2);
		}

		return 0;
	}

	public function getGiftMilestones(): array
	{
		$cartItems = $this->getCartItemsForGifts();
		if (!$cartItems) {
			return [];
		}

		$milestones = [];
		$gifts      = $this->getGifts();
		if (!$gifts) {
			return [];
		}

		$itemsPrice = 0;
		foreach ($cartItems as $item) {
			$itemsPrice += $item->getTotalPrice();
		}

		usort($gifts, static fn(Gift $a, Gift $b) => $a->priceFrom <=> $b->priceFrom);
		foreach ($gifts as $gift) {
			if ($gift->priceFrom === null) {
				continue;
			}

			$milestones[(string) $gift->priceFrom]['items'][] = $gift;
		}

		$currentGift = $this->getOrderGift();
		foreach ($milestones as $priceFromKey => $milestone) {
			$isSelected   = false;
			$selectedGift = null;
			$priceFrom    = Caster::toFloat($priceFromKey);

			if ($currentGift) {
				foreach ($milestone['items'] as $item) {
					if ($item->id === $currentGift->id) {
						$isSelected   = true;
						$selectedGift = $item;
					}
				}
			}

			$percentValue                              = $priceFrom > 0 ? ($itemsPrice / $priceFrom) * 100 : 0;
			$remaining                                 = (float) max($priceFrom - $itemsPrice, 0);
			$milestones[$priceFromKey]['percentValue'] = min($percentValue, 100.0);
			$milestones[$priceFromKey]['isSelectable'] = Floats::isGreaterThanOrEqualTo($itemsPrice, $priceFrom);
			$milestones[$priceFromKey]['priceFrom']    = $priceFrom;
			$milestones[$priceFromKey]['priceItems']   = $itemsPrice;
			$milestones[$priceFromKey]['remaining']    = $remaining;
			$milestones[$priceFromKey]['isSelected']   = $isSelected;
			$milestones[$priceFromKey]['selectedGift'] = $selectedGift;
		}

		return array_values($milestones);
	}

	public function isEnabledGiftMilestones(): bool
	{
		if (!Parameters::load('eshopGifts.front.milestones')) {
			return false;
		}

		return !Arrays::every($this->getCartItems(), static fn(CartItem $ci) => $ci->getProduct() && $ci->getProduct()->isDiscount);
	}

}
