<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Model;

use Contributte\Translation\Translator;
use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Helpers\Arrays;
use Core\Model\Sites;
use DateTime;
use Doctrine\DBAL\ParameterType;
use EshopOrders\FrontModule\Model\Dao\CustomerLoyalty;
use EshopOrders\Model\Cache\LoyaltyCache;
use EshopOrders\Model\CacheService;
use EshopOrders\Model\Loyalty\LoyaltyCalculator;
use EshopOrders\Model\Loyalty\LoyaltyPointsManager;
use EshopOrders\Model\Loyalty\LoyaltySettings;
use Nette\Caching\Cache;

class LoyaltyService
{
	public function __construct(
		protected EntityManagerDecorator $em,
		protected Sites                  $sites,
		protected LoyaltyCalculator      $loyaltyCalculator,
		protected LoyaltySettings        $loyaltySettings,
		protected CacheService           $cacheService,
		protected LoyaltyPointsManager   $loyaltyPointsManager,
		protected Translator             $translator,
	)
	{
	}

	public function getCustomerDao(int $customerId): ?CustomerLoyalty
	{
		$siteIdent = $this->sites->getCurrentSite()->getIdent();
		$cacheKey  = LoyaltyCache::getCustomerKey($customerId, $siteIdent);

		/** @var CustomerLoyalty|null $dao */
		$dao = $this->cacheService->loyaltyCache->cache->load($cacheKey, function(&$dep) use ($customerId, $siteIdent) {
			$dep = [
				Cache::Expire => '1 month',
				Cache::Tags   => [LoyaltyCache::cacheKeyCustomer],
			];

			$settings = $this->loyaltySettings->getSettings($siteIdent);
			if (!$settings) {
				return null;
			}

			$data = $this->em->getConnection()->fetchAssociative("SELECT points, last_activity_at, email_expire_sent 
					FROM eshop_orders__customer_loyalty 
					WHERE customer_id = :customer AND site_id = :siteIdent
					LIMIT 1", [
				'customer'  => $customerId,
				'siteIdent' => $siteIdent,
			], [
				'customer'  => ParameterType::INTEGER,
				'siteIdent' => ParameterType::STRING,
			]);

			if (!$data) {
				return null;
			}

			$lastActivity    = $data['last_activity_at'] ? DateTime::createFromFormat('Y-m-d H:i:s', $data['last_activity_at']) : null;
			$emailExpireSent = $data['email_expire_sent'] ? DateTime::createFromFormat('Y-m-d H:i:s', $data['email_expire_sent']) : null;

			return new CustomerLoyalty(
				customerId     : $customerId,
				points         : $data['points'] ?? 0,
				icon           : $settings->icon,
				lastActivityAt : $lastActivity ?: null,
				emailExpireSent: $emailExpireSent ?: null,
			);
		});

		return $dao;
	}

	public function calculatePoints(float $price): int
	{
		$siteIdent = $this->sites->getCurrentSite()->getIdent();

		return $this->loyaltyCalculator->calculate($price, $siteIdent);
	}

	public function getExpireDate(int $customerId): ?DateTime
	{
		$siteIdent = $this->sites->getCurrentSite()->getIdent();
		$settings  = $this->loyaltySettings->getSettings($siteIdent);

		if (!$settings) {
			return null;
		}

		$customerDao = $this->getCustomerDao($customerId);

		$daysToExpire = $settings->pointsExpire;
		$lastActivity = $customerDao->lastActivityAt;

		if (!$lastActivity) {
			return null;
		}

		return $lastActivity->modify("+{$daysToExpire} days");
	}

	public function findLastPointsHistory(int $customerId, int $limit = 10): array
	{
		$siteIdent = $this->sites->getCurrentSite()->getIdent();
		$settings  = $this->loyaltySettings->getSettings($siteIdent);

		if (!$settings) {
			return [];
		}

		$result    = [];
		$loyaltyId = $this->loyaltyPointsManager->getLoyaltyId($customerId, $siteIdent);

		if ($loyaltyId) {
			foreach ($this->em->getConnection()
				         ->iterateAssociative("SELECT points, reason, reference_id, created_at 
								FROM eshop_orders__customer_loyalty_action 
								WHERE customer_loyalty_id = :loyaltyId 
								ORDER BY created_at DESC LIMIT :limit", [
					         'loyaltyId' => $loyaltyId,
					         'limit'     => $limit,
				         ]) as $row) {
				/** @var array $row */
				$created = DateTime::createFromFormat('Y-m-d H:i:s', $row['created_at']) ?: new DateTime();

				$result[] = [
					'points'    => $row['points'],
					'reason'    => $this->translateReason($row['reason']),
					'orderId'   => $row['reference_id'],
					'createdAt' => $created,
				];
			}
		}

		return $result;
	}

	public function translateReason(string $reason): string
	{
		if (Arrays::contains([
			LoyaltyPointsManager::reasonUse,
			LoyaltyPointsManager::reasonOrder,
			LoyaltyPointsManager::reasonOrderCancel,
		], $reason)) {
			return $this->translator->translate('eshopOrdersFront.loyalty.reasonList.' . $reason);
		}

		return $reason;
	}
}
