<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Presenters;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Event\Event;
use Core\Model\Helpers\Arrays;
use EshopCatalog\FrontModule\Components\CartAddForm;
use EshopCatalog\FrontModule\Components\ICartAddFormFactory;
use EshopCatalog\FrontModule\Components\IProductPreviewFactory;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopCatalog\Model\Config;
use EshopOrders\FrontModule\Components\Customer\AccountSettings;
use EshopOrders\FrontModule\Components\Customer\AddressesList;
use EshopOrders\FrontModule\Components\Customer\ContactData;
use EshopOrders\FrontModule\Components\Customer\IAccountSettingsFactory;
use EshopOrders\FrontModule\Components\Customer\IAddressesListFactory;
use EshopOrders\FrontModule\Components\Customer\IContactDataFactory;
use EshopOrders\FrontModule\Components\Customer\IOrdersListFactory;
use EshopOrders\FrontModule\Components\Customer\IProfileFormFactory;
use EshopOrders\FrontModule\Components\Customer\IPurchasedProductsFactory;
use EshopOrders\FrontModule\Components\Customer\OrdersList;
use EshopOrders\FrontModule\Components\Customer\ProfileForm;
use EshopOrders\FrontModule\Components\Customer\PurchasedProducts;
use EshopOrders\FrontModule\Model\Customers;
use EshopOrders\FrontModule\Model\LoyaltyService;
use EshopOrders\FrontModule\Model\OrderFacade;
use EshopOrders\FrontModule\Model\Orders;
use EshopOrders\FrontModule\Model\ProductsAssignedToCustomers;
use EshopOrders\Model\Entities\Customer;
use EshopOrders\Model\Entities\Invoice;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\EshopOrdersConfig;
use EshopOrders\Model\Helpers\OrderHelper;
use EshopOrders\Model\Invoices;
use EshopOrders\Model\Pdf\IInvoicePdfBuilderFactory;
use Nette\Application\BadRequestException;
use Nette\Application\Responses\FileResponse;
use Nette\Application\UI\Multiplier;
use Nette\Utils\Html;
use Override;
use EshopOrders\FrontModule\Components\Customer\Loyalty\IPointsHistoryFactory;
use EshopOrders\FrontModule\Components\Customer\Loyalty\PointsHistory;
use Users\FrontModule\Components\IOAuthUserConnectFactory;
use Users\FrontModule\Components\OAuthUserConnect;
use Users\FrontModule\Components\User\UserGridMenu;
use Users\FrontModule\Components\User\UserSide;
use Users\Model\Entities\User;
use Users\Model\Users;

class CustomerPresenter extends BasePresenter
{
	public ?Customer $currentCustomer = null;
	protected array  $cOrders         = [];

	public function __construct(
		protected Orders                      $ordersService,
		protected Users                       $usersService,
		protected Customers                   $customersService,
		public Invoices                       $invoices,
		public IInvoicePdfBuilderFactory      $invoicePdfBuilderFactory,
		protected ProductsAssignedToCustomers $productsAssignedToCustomers,
		protected ProductsFacade              $productsFacade,
		protected EntityManagerDecorator      $em,
		protected OrderFacade                 $orderFacade,
		protected LoyaltyService              $loyaltyService,
	)
	{
		parent::__construct();
	}

	#[Override]
	public function checkRequirements($element): void
	{
		parent::checkRequirements($element);
		if (!$this->getUser()->isLoggedIn()) {
			$this->redirect(':Users:Front:Login:');
		}
	}

	#[Override]
	protected function startup()
	{
		parent::startup();

		/** @var User $user */
		$user = $this->getUser()->getIdentity();

		$this->template->customer = null;

		if ($user) {
			$this->currentCustomer = $this->customersService->getByUser($user);

			if (!$this->currentCustomer) {
				$this->currentCustomer = $this->customersService->getOrCreateCustomer($user->getId(), $user->getEmail(), $user->getName(), $user->getLastname(), $user->phone);
			}

			$this->template->currentCustomer = $this->currentCustomer;
		}

		$this['meta']->setMeta('robots', 'noindex, nofollow');

		/** @var UserSide $userSide */
		$userSide = $this->getComponent('userSide');
		$userSide->setMenuItem('eshopOrders-orders', '_eshopOrdersFront.customerMenu.orders', [':EshopOrders:Front:Customer:orders']);

		if (EshopOrdersConfig::load('customerAccount.allowPurchasedProducts')) {
			$userSide->setMenuItem('eshopOrders-purchasedProducts', '_eshopOrdersFront.customerMenu.purchasedProducts', [':EshopOrders:Front:Customer:purchasedProducts']);
		}

		$userSide->setMenuItem('eshopOrders-accountSettings', '_eshopOrdersFront.customerMenu.accountSettings', [':EshopOrders:Front:Customer:accountSettings']);
		$userSide->setMenuItem('eshopOrders-contactData', '_eshopOrdersFront.customerMenu.contactData', [':EshopOrders:Front:Customer:contactData']);

		if (Config::load('product.allowAssignProductToCustomer')) {
			$hasMyProducts                 = (bool) $this->productsAssignedToCustomers->getProductsIdByUser((int) $this->getUser()->getId());
			$this->template->hasMyProducts = $hasMyProducts;

			if ($hasMyProducts) {
				$userSide->setMenuItem('eshopOrders-myProducts', '_eshopOrdersFront.customerMenu.myProducts', [':EshopOrders:Front:Customer:myProducts']);
			}
		}

		$loyaltyDao = $this->loyaltyService->getCustomerDao($this->currentCustomer->getId());
		if ($loyaltyDao) {
			$userSide->addSideContent('loyaltyPoints', Html::el('div', [
				'class' => 'users-user__side-loyalty-points',
			])
				->addHtml(Html::el('span')->setText($this->t('eshopOrdersFront.loyalty.userSide.loyaltyPoints')))
				->addHtml(Html::el('a', [
					'href' => $this->link(':EshopOrders:Front:Customer:loyaltyPoints'),
				])->setText(number_format($loyaltyDao->points, 0, '', ' ')))
				->addHtml(Html::el('img', [
					'src' => $loyaltyDao->icon,
				])),
				0
			);
		}
	}

	public function handleOrderAttachment(int $orderId, string $filename): void
	{
		$order = $this->ordersService->getDetail($orderId);

		if (!$order || !$order->getCustomer()) {
			throw new BadRequestException();
		}

		/** @var User $user */
		$user              = $this->getUser()->getIdentity();
		$customer          = $order->getCustomer();
		$parentViewAllowed = false;

		if (
			EshopOrdersConfig::load('showOrdersOfChildrenUsers')
			&& (
				$user->childs->containsKey($order->getCustomer()->getUser()->getId())
				|| ($user->getParam('childUsers') && Arrays::contains($user->getParam('childUsers'), $order->getCustomer()->getUser()->getId()))
			)
		) {
			$parentViewAllowed = true;
		}

		if ($order->site->getIdent() !== $this->sitesService->getCurrentSite()->getIdent()
			|| ($order->getCustomer()->getId() !== $customer->getId() && !$parentViewAllowed)
		) {
			throw new BadRequestException();
		}

		$filePath = OrderHelper::getOrderAttachmentDir($orderId) . '/' . $filename;
		if (!file_exists($filePath)) {
			throw new BadRequestException;
		}

		$contentType = mime_content_type($filePath) ?: null;

		$this->sendResponse(new FileResponse($filePath, basename($filePath), $contentType, false));
	}

	public function handleRepeatOrder(int $orderId): void
	{
		$result = $this->orderFacade->repeatOrder($orderId);
		if ($result['added'] === 0) {
			$this->flashMessageDanger('eshopOrdersFront.customer.repeatOrderNoProducts');
		} else if ($result['notQuantity'] > 0) {
			$this->flashMessageInfo('eshopOrdersFront.customer.repeatOrderNoQuantity');
		}

		if ($result['status'] === 'ok') {
			$this->flashMessageSuccess('eshopOrdersFront.customer.repeatOrderSuccess');

			$event = new Event(['orderId' => $orderId]);
			$this->eventDispatcher->dispatch($event, 'eshopOrders.repeatOrder');

			$this->redirect(':EshopOrders:Front:Default:order');
		} else {
			$this->flashMessageDanger('eshopOrdersFront.customer.repeatOrderError');

			$this->redirect('this');
		}
	}

	/*******************************************************************************************************************
	 * ============================== Actions
	 */

	public function actionDefault(): void
	{
		$lastOrderCount = (int) EshopOrdersConfig::load('customerAccount.lastOrdersCount');

		/** @var Order[] $lastOrders */
		$lastOrders = $this->em->getRepository(Order::class)->createQueryBuilder('o')
			->where('o.customer = :customer')
			->andWhere('o.site = :site')
			->setParameter('customer', $this->currentCustomer->getId())
			->setParameter('site', $this->sitesService->getCurrentSite()->getIdent())
			->orderBy('o.id', 'DESC')
			->setMaxResults($lastOrderCount)
			->getQuery()->getResult();

		foreach ($lastOrders as $v) {
			$this->cOrders[$v->getId()] = $v;
		}

		$this->template->lastOrders   = $lastOrders;
		$this->template->messages     = $this->getTemplate()->flashes;
		$this->getTemplate()->flashes = [];
	}

	public function actionOrderDetail(int $orderId): void
	{
		$order = $this->ordersService->getDetail($orderId);

		if (!$order instanceof Order) {
			throw new BadRequestException();
		}

		$customer = $this->currentCustomer;
		$user     = $customer->getUser();

		$parentViewAllowed = false;
		if (
			EshopOrdersConfig::load('showOrdersOfChildrenUsers')
			&& (
				$user->childs->containsKey($order->getCustomer()->getUser()->getId())
				|| ($user->getParam('childUsers') && Arrays::contains($user->getParam('childUsers'), $order->getCustomer()->getUser()->getId()))
			)
		) {
			$parentViewAllowed = true;
		}

		if (
			!$order->getCustomer()
			|| $order->site->getIdent() !== $this->sitesService->getCurrentSite()->getIdent()
			|| ($order->getCustomer()->getId() !== $customer->getId() && !$parentViewAllowed)
		) {
			throw new BadRequestException();
		}

		$tmp = [];
		foreach ($order->getOrderItems() as $item) {
			if ($item->getProductId()) {
				$tmp[] = $item->getProductId();
			}
		}

		$tmp = array_unique($tmp);

		$products = $tmp === [] ? [] : $this->productsFacade->getProducts($tmp);
		$this->productsFacade->productsService->loadDocuments($products);

		$this->template->showUser = EshopOrdersConfig::load('showOrdersOfChildrenUsers');
		$this->template->products = $products;
		$this->template->order    = $order;
	}

	public function actionMyProducts(): void
	{
		if (!Config::load('product.allowAssignProductToCustomer')) {
			$this->error();
		}

		$ids = $this->productsAssignedToCustomers->getProductsIdByUser((int) $this->getUser()->getId());

		if (!empty($ids)) {
			$this->template->products = $this->productsFacade->getProducts($ids);
		} else {
			$this->template->products = [];
		}

		/** @var UserSide $userSide */
		$userSide = $this->getComponent('userSide');
		$userSide->setActiveItem('eshopOrders-myProducts');
	}

	public function actionPurchasedProducts(): void
	{
		if (!EshopOrdersConfig::load('customerAccount.allowPurchasedProducts')) {
			$this->error();
		}

		/** @var UserSide $userSide */
		$userSide = $this->getComponent('userSide');
		$userSide->setActiveItem('eshopOrders-purchasedProducts');
	}

	public function actionEditProfile(): void
	{
	}

	public function actionInvoice(int $id): void
	{
		if (!EshopOrdersConfig::load('invoice.enable', false)) {
			throw new BadRequestException();
		}

		/** @var Order $order */
		$order = $this->ordersService->getDetail($id);

		/** @var User $user */
		$user = $this->getUser()->getIdentity();

		$customer = $this->currentCustomer;

		$parentViewAllowed = false;
		if (
			EshopOrdersConfig::load('showOrdersOfChildrenUsers')
			&& (
				$user->childs->containsKey($order->getCustomer()->getUser()->getId())
				|| ($user->getParam('childUsers') && Arrays::contains($user->getParam('childUsers'), $order->getCustomer()->getUser()->getId()))
			)
		) {
			$parentViewAllowed = true;
		}

		if (!$order || !$order->getCustomer() || !$customer || ($order->getCustomer()->getId() != $customer->getId() && !$parentViewAllowed)) {
			throw new BadRequestException();
		}

		$invoice = $this->invoices->getOneByOrder($order, false);

		if (!$invoice instanceof Invoice) {
			throw new BadRequestException();
		}

		$this->invoicePdfBuilderFactory->create($invoice)->render();
	}

	public function actionLoyaltyPoints(): void
	{
		if (!EshopOrdersConfig::load('allowLoyaltyPoints')) {
			$this->error();
		}

		$this->template->pointsExpireDate = $this->loyaltyService->getExpireDate($this->currentCustomer->getId());

		$activeNavigation = $this->getActiveNavigation();
		$this->setPageTitle($activeNavigation->getPageTitle() . ' - ' . $this->t('eshopOrdersFront.loyalty.userSide.pageTitle'));
	}

	public function renderDefault(): void
	{
		/** @var UserSide $userSide */
		$userSide = $this->getComponent('userSide');
		$userSide->setActiveItem('home');

		/** @var UserGridMenu $userSide */
		$userSide = $this->getComponent('userGridMenu');
		$userSide->setMenuItem('eshopOrders-orders', '_eshopOrdersFront.customerMenu.orders', SRC_DIR . '/core/assets/img/assignment_turned_in.svg', [':EshopOrders:Front:Customer:orders']);

		if (EshopOrdersConfig::load('customerAccount.allowPurchasedProducts')) {
			$userSide->setMenuItem('eshopOrders-purchasedProducts', '_eshopOrdersFront.customerMenu.purchasedProducts', SRC_DIR . '/core/assets/img/purchased-products.svg', [':EshopOrders:Front:Customer:purchasedProducts']);
		}

		$userSide->setMenuItem('eshopOrders-accountSettings', '_eshopOrdersFront.customerMenu.accountSettings', SRC_DIR . '/core/assets/img/settings.svg', [':EshopOrders:Front:Customer:accountSettings']);
		$userSide->setMenuItem('eshopOrders-contactData', '_eshopOrdersFront.customerMenu.contactData', SRC_DIR . '/core/assets/img/article_person.svg', [':EshopOrders:Front:Customer:contactData']);
	}

	public function renderOrders(): void
	{
		/** @var UserSide $userSide */
		$userSide = $this->getComponent('userSide');
		$userSide->setActiveItem('eshopOrders-orders');
	}

	public function renderOrderDetail(): void
	{
		/** @var UserSide $userSide */
		$userSide = $this->getComponent('userSide');
		$userSide->setActiveItem('eshopOrders-orders');
	}

	public function renderContactData(): void
	{
		/** @var UserSide $userSide */
		$userSide = $this->getComponent('userSide');
		$userSide->setActiveItem('eshopOrders-contactData');
	}

	public function renderAccountSettings(): void
	{
		/** @var UserSide $userSide */
		$userSide = $this->getComponent('userSide');
		$userSide->setActiveItem('eshopOrders-accountSettings');
	}

	/*******************************************************************************************************************
	 * ============================== Components
	 */

	protected function createComponentOrdersList(IOrdersListFactory $factory): OrdersList
	{
		return $factory->create();
	}

	protected function createComponentPurchasedProducts(IPurchasedProductsFactory $factory): PurchasedProducts
	{
		return $factory->create();
	}

	protected function createComponentProfileForm(IProfileFormFactory $factory): ProfileForm
	{
		/** @var User $user */
		$user    = $this->getUser()->getIdentity();
		$control = $factory->create($user);

		$control['form']->onSuccess['redirect'] = function(): void {
			$this->redirect('this');
		};

		return $control;
	}

	protected function createComponentContactData(IContactDataFactory $factory): ContactData
	{
		/** @var User $user */
		$user = $this->getUser()->getIdentity();

		return $factory->create($user);
	}

	protected function createComponentAccountSettings(IAccountSettingsFactory $factory): AccountSettings
	{
		/** @var User $user */
		$user = $this->getUser()->getIdentity();

		return $factory->create($user);
	}

	protected function createComponentAddressesList(IAddressesListFactory $factory): AddressesList
	{
		$control = $factory->create((int) $this->getUser()->getId());

		$control->onAddressSave[] = function(AddressesList $control): void {
			$this['profileForm']->redrawControl('formAddresses');
			$this['profileForm']->redrawControl('formAddresses2');
		};

		return $control;
	}

	protected function createComponentProductPreview(IProductPreviewFactory $factory): Multiplier
	{
		return new Multiplier(static function($id) use ($factory) {
			$control = $factory->create();
			$control->setProductById((int) $id);

			return $control;
		});
	}

	protected function createComponentCartAddForm(ICartAddFormFactory $factory): Multiplier
	{
		return new Multiplier(function($id) use ($factory) {
			$control           = $factory->create();
			$control->viewType = CartAddForm::VIEWTYPE_CUSTOMER;
			$control->setProduct($this->productsFacade->getProduct((int) $id));

			return $control;
		});
	}

	protected function createComponentOauthUserConnect(IOAuthUserConnectFactory $factory): OAuthUserConnect
	{
		return $factory->create();
	}

	protected function createComponentLoyaltyPointsHistory(IPointsHistoryFactory $factory): PointsHistory
	{
		return $factory->create($this->currentCustomer);
	}
}
