<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Presenters;

use EshopCatalog\DI\EshopCatalogExtension;
use EshopCatalog\FrontModule\Components\CartAddForm;
use EshopCatalog\FrontModule\Components\ICartAddFormFactory;
use EshopCatalog\FrontModule\Helpers\TRecentlyViewedProducts;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopCatalog\Model\Config as EshopCatalogConfig;
use EshopOrders\FrontModule\Components\Cart\CartDetail;
use EshopOrders\FrontModule\Components\Cart\ICartDetailFactory;
use EshopOrders\FrontModule\Components\Order\IOrderFormFactory;
use EshopOrders\FrontModule\Components\Order\OrderForm;
use EshopOrders\FrontModule\Helpers\CustomerHelper;
use EshopOrders\FrontModule\Model\Carts;
use EshopOrders\FrontModule\Model\Customers;
use EshopOrders\Model\Orders;
use Nette\Application\UI\Multiplier;
use Users\Model\Entities\User;

class DefaultPresenter extends BasePresenter
{
	use TRecentlyViewedProducts;

	public function __construct(
		protected Orders         $ordersService,
		public Carts             $cartsService,
		protected ProductsFacade $productsFacade,
		protected Customers      $customersService
	)
	{
		parent::__construct();
	}

	public function cartUpdateListener(): void
	{
		$this->redrawControl('orderForm');

		/** @var User|null $user */
		$user                   = $this->getUser()->isLoggedIn() ? $this->getUser()->getIdentity() : null;
		$quantityToOrderMissing = 0;

		if ($user) {
			$customer = $this->customersService->getByUser($user);
			$cart     = $this->cartsService->getCurrentCart();

			if ($customer && $cart) {
				$quantityToOrderMissing = CustomerHelper::checkCustomerMultipleQuantitiesInCart($customer, $cart);
			}
		}

		$this->template->quantityToOrderMissing = $quantityToOrderMissing;

		$this['cartDetail']->showCartSteps = $quantityToOrderMissing === 0;

		$this['cartDetail']->redrawControl('cartSteps');
		$this->redrawControl('cartMessage');
	}

	/*******************************************************************************************************************
	 * ============================== Actions
	 */

	public function actionOrder(): void
	{
		$this['meta']->setMeta('robots', 'noindex, nofollow');

		if (class_exists(EshopCatalogExtension::class)
			&& EshopCatalogConfig::load('allowAddToCartOnlyForLoggedUsers') && !$this->getUser()->isLoggedIn()
			&& is_string($this->getHomepage()->link)) {
			$this->redirectUrl($this->getHomepage()->link);
		}

		/** @var User|null $user */
		$user  = $this->getUser()->isLoggedIn() ? $this->getUser()->getIdentity() : null;
		$group = null;

		$quantityToOrderMissing = 0;
		if ($user) {
			$customer = $this->customersService->getByUser($user);
			$cart     = $this->cartsService->getCurrentCart();

			if ($customer && $cart) {
				$quantityToOrderMissing = CustomerHelper::checkCustomerMultipleQuantitiesInCart($customer, $cart);
				$group                  = $customer->getGroupCustomers();
			}
		}

		$this['cartDetail']->showCartSteps = $quantityToOrderMissing === 0;

		$this->template->customerGroup          = $group;
		$this->template->quantityToOrderMissing = $quantityToOrderMissing;
	}

	/*******************************************************************************************************************
	 * ============================== Render
	 */

	public function renderOrder(): void
	{
		$cart                        = $this->cartsService->getCurrentCart();
		$this->template->pageClass   .= ' page-cart';
		$this->template->cartIsEmpty = $cart->getCartItems() === [];
	}

	/*******************************************************************************************************************
	 * ============================== Components
	 */

	protected function createComponentOrderForm(IOrderFormFactory $factory): OrderForm
	{
		return $factory->create();
	}

	protected function createComponentCartDetail(ICartDetailFactory $factory): CartDetail
	{
		$control = $factory->create();

		$this->eventDispatcher->addListener('eshopOrders.cartUpdateItem', $this->cartUpdateListener(...));
		$this->eventDispatcher->addListener('eshopOrders.cartRemoveItem', $this->cartUpdateListener(...));

		return $control;
	}

	protected function createComponentCartAddForm(ICartAddFormFactory $factory): Multiplier
	{
		return new Multiplier(function($productId) use ($factory): CartAddForm {
			$control = $factory->create();
			$control->setProduct($this->productsFacade->getProduct((int) $productId));
			$control['form']->onSuccess[] = function(): void {
				$this->redrawControl('cart');
			};

			return $control;
		});
	}
}
