<?php declare(strict_types = 1);

namespace EshopOrders\Model\Entities;

use Core\Model\Entities\TId;
use Core\Model\Helpers\Traits\TExtraField;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\Mapping as ORM;
use EshopOrders\Model\EshopOrdersConfig;
use EshopOrders\Model\Listeners\UserListener;
use Users\Model\Entities\User;

#[ORM\Table('eshop_orders__customer')]
#[ORM\Entity]
#[ORM\EntityListeners([UserListener::class])]
class Customer
{
	use TId;
	use TExtraField;

	public const EXTRA_FIELD_SECTION = 'eshopOrdersCustomer';

	public const WARNING_INVOICE_REMINDER = 1, DANGER_INVOICE_REMINDER = 2;

	public static array $invoiceReminderOptions = [self::WARNING_INVOICE_REMINDER, self::DANGER_INVOICE_REMINDER];

	#[ORM\JoinColumn(name: 'user_id', referencedColumnName: 'id', onDelete: 'CASCADE')]
	#[ORM\OneToOne(targetEntity: User::class)]
	public User $user;

	#[ORM\Column(name: 'phone', type: Types::STRING, nullable: true)]
	protected ?string $phone = null;

	#[ORM\Column(type: Types::SMALLINT, options: ['default' => 0])]
	public int $isBlacklisted = 0;

	#[ORM\Column(type: Types::INTEGER, options: ['default' => 0])]
	public int $invoiceReminderCount = 0;

	/** defaultni (prave vybrana) dorucovaci adresa
	 */
	#[ORM\JoinColumn(name: 'address_delivery', referencedColumnName: 'id', nullable: true, onDelete: 'SET NULL')]
	#[ORM\ManyToOne(targetEntity: CustomerAddress::class)]
	public ?CustomerAddress $addressDelivery = null;

	/** defaultni (prave vybrana) dorucovaci adresa
	 */
	#[ORM\JoinColumn(name: 'address_invoice', referencedColumnName: 'id', nullable: true, onDelete: 'SET NULL')]
	#[ORM\ManyToOne(targetEntity: CustomerAddress::class)]
	public ?CustomerAddress $addressInvoice = null;

	/** @var Collection<CustomerAddress> */
	#[ORM\JoinTable(name: 'eshop_orders__customer_customer_address')]
	#[ORM\JoinColumn(name: 'customer_id', referencedColumnName: 'id', onDelete: 'CASCADE')]
	#[ORM\InverseJoinColumn(name: 'address_id', referencedColumnName: 'id', onDelete: 'CASCADE')]
	#[ORM\ManyToMany(targetEntity: CustomerAddress::class, inversedBy: 'customers', indexBy: 'id')]
	public Collection $addressesAvailable;

	/** @var Collection<Order> */
	#[ORM\OneToMany(mappedBy: 'customer', targetEntity: Order::class, indexBy: 'id')]
	public Collection $orders;

	#[ORM\JoinColumn(name: 'id_group_customers', referencedColumnName: 'id', nullable: true, onDelete: 'SET NULL')]
	#[ORM\ManyToOne(targetEntity: GroupCustomers::class, cascade: ['persist'])]
	protected ?GroupCustomers $groupCustomers = null;

	/** @var Collection<Payment> */
	#[ORM\JoinTable(name: 'eshop_orders__customer_disabled_payment')]
	#[ORM\JoinColumn(name: 'customer_id', referencedColumnName: 'id', onDelete: 'CASCADE')]
	#[ORM\InverseJoinColumn(name: 'payment_id', referencedColumnName: 'id', onDelete: 'CASCADE')]
	#[ORM\ManyToMany(targetEntity: Payment::class, indexBy: 'id')]
	public Collection $disabledPayments;

	/** @var Collection<Spedition> */
	#[ORM\JoinTable(name: 'eshop_orders__customer_disabled_spedition')]
	#[ORM\JoinColumn(name: 'customer_id', referencedColumnName: 'id', onDelete: 'CASCADE')]
	#[ORM\InverseJoinColumn(name: 'spedition_id', referencedColumnName: 'id', onDelete: 'CASCADE')]
	#[ORM\ManyToMany(targetEntity: Spedition::class, indexBy: 'id')]
	public Collection $disabledSpeditions;

	public function __construct(
		User    $user,
		?string $phone
	)
	{
		$this->user               = $user;
		$this->phone              = $phone;
		$this->addressesAvailable = new ArrayCollection();
		$this->orders             = new ArrayCollection();
		$this->disabledPayments   = new ArrayCollection();
		$this->disabledSpeditions = new ArrayCollection();
	}

	public function getUser(): User
	{
		return $this->user;
	}

	public function getPhone(): string
	{
		return $this->phone ?: "";
	}

	public function setPhone(string $phone): Customer
	{
		$this->phone = $phone;

		return $this;
	}

	public function getAddressDelivery(): ?CustomerAddress { return $this->addressDelivery; }

	public function setAddressDelivery(?CustomerAddress $addressDelivery): Customer
	{
		$this->addressDelivery = $addressDelivery;
		if ($addressDelivery && !$this->addressesAvailable->contains($addressDelivery)) {
			$this->addressesAvailable->add($addressDelivery);
		}

		return $this;
	}

	public function getAddressInvoice(): ?CustomerAddress { return $this->addressInvoice; }

	public function setAddressInvoice(?CustomerAddress $addressInvoice): Customer
	{
		$this->addressInvoice = $addressInvoice;
		if ($addressInvoice && !$this->addressesAvailable->contains($addressInvoice)) {
			$this->addressesAvailable->add($addressInvoice);
		}

		return $this;
	}

	/**
	 * @return Collection<Order>
	 */
	public function getOrders() { return $this->orders; }

	/** TODO pridavat a ubirat po jedne?
	 *
	 * @param ArrayCollection|Order[] $orders
	 */
	public function setOrders($orders): self
	{
		$this->orders = $orders;

		return $this;
	}

	public function getGroupCustomers(): ?GroupCustomers { return $this->groupCustomers; }

	public function setGroupCustomers(GroupCustomers $groupCustomers): void { $this->groupCustomers = $groupCustomers; }

	public function removeGroupCustomers(): void { $this->groupCustomers = null; }

	public function hasMinimalOrderPrice(): float
	{
		if (EshopOrdersConfig::load('customer.allowMinimalOrderPrice', false)) {
			$ef = $this->getExtraFieldsValues();

			if (isset($ef['allowMinimalOrderPrice']) && $ef['allowMinimalOrderPrice'] == true && $ef['minimalOrderPrice'] > 0) {
				return (float) str_replace(',', '.', (string) $ef['minimalOrderPrice']);
			}
		}

		return 0;
	}
}

