<?php declare(strict_types = 1);

namespace EshopOrders\Model\Entities;

use Core\Model\Entities\Country;
use Core\Model\Entities\TId;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\Mapping as ORM;
use EshopOrders\Model\EshopOrdersConfig;
use EshopOrders\Model\Listeners\UserListener;

#[ORM\Table('eshop_orders__customer_address')]
#[ORM\Entity]
#[ORM\EntityListeners([UserListener::class])]
class CustomerAddress
{
	use TId;

	#[ORM\Column(name: 'address_name', type: Types::STRING, length: 255, nullable: true)]
	public ?string $addressName = null;

	#[ORM\Column(name: 'first_name', type: Types::STRING, nullable: true)]
	public ?string $firstName = null;

	#[ORM\Column(name: 'last_name', type: Types::STRING, nullable: true)]
	public ?string $lastName = null;

	#[ORM\Column(name: 'email', type: Types::STRING, length: 255, nullable: true)]
	public ?string $email = null;

	#[ORM\Column(name: 'phone', type: Types::STRING, length: 255, nullable: true)]
	public ?string $phone = null;

	#[ORM\Column(name: 'company', type: Types::STRING, nullable: true)]
	public ?string $company = null;

	#[ORM\Column(name: 'street', type: Types::STRING, nullable: true)]
	public ?string $street = null;

	#[ORM\Column(name: 'city', type: Types::STRING, nullable: true)]
	public ?string $city = null;

	#[ORM\Column(name: 'postal', type: Types::STRING, nullable: true)]
	public ?string $postal = null;

	#[ORM\Column(name: 'id_number', type: Types::STRING, nullable: true)]
	public ?string $idNumber = null;

	#[ORM\Column(name: 'vat_number', type: Types::STRING, nullable: true)]
	public ?string $vatNumber = null;

	#[ORM\Column(name: 'id_vat_number', type: Types::STRING, length: 255, nullable: true)]
	public ?string $idVatNumber = null;

	#[ORM\Column(name: 'customer_id', type: Types::INTEGER, nullable: true, options: ['unsigned' => true])]
	protected ?int $customer = null;

	#[ORM\JoinColumn(name: 'country_id', referencedColumnName: 'id', onDelete: 'SET NULL')]
	#[ORM\ManyToOne(targetEntity: Country::class)]
	public ?Country $country = null;

	#[ORM\Column(name: 'validated_id_number', type: Types::SMALLINT, length: 1, options: ['default' => 0])]
	public int $validatedIdNumber = 0;

	#[ORM\Column(name: 'validated_vat_number', type: Types::SMALLINT, length: 1, options: ['default' => 0])]
	public int $validatedVatNumber = 0;

	#[ORM\Column(name: 'params', type: Types::JSON, nullable: true)]
	public ?array $params = null;

	/** @var Collection<Customer> */
	#[ORM\ManyToMany(targetEntity: Customer::class, mappedBy: 'addressesAvailable', indexBy: 'id')]
	public Collection $customers;

	public function __construct(Customer $customer)
	{
		$customer->addressesAvailable->add($this);
		$this->customers = new ArrayCollection();
	}

	public function getFirstName(): ?string { return $this->firstName; }

	public function setFirstName(?string $firstName): self
	{
		$this->firstName = $firstName;

		return $this;
	}

	public function getLastName(): ?string { return $this->lastName; }

	public function setLastName(?string $lastName): self
	{
		$this->lastName = $lastName;

		return $this;
	}

	public function getName(): string { return trim($this->getFirstName() . ' ' . $this->getLastName()); }

	public function getCompany(): ?string { return $this->company; }

	public function setCompany(?string $company): self
	{
		$this->company = $company;

		return $this;
	}

	public function getStreet(): ?string { return $this->street; }

	public function setStreet(?string $street): self
	{
		$this->street = $street;

		return $this;
	}

	public function getCity(): ?string { return $this->city; }

	public function setCity(?string $city): self
	{
		$this->city = $city;

		return $this;
	}

	public function getPostal(): ?string { return $this->postal; }

	public function setPostal(?string $postal): self
	{
		$this->postal = $postal;

		return $this;
	}

	public function getIdNumber(): ?string { return $this->idNumber; }

	public function setIdNumber(?string $idNumber): self
	{
		$this->idNumber = $idNumber;

		return $this;
	}

	public function getVatNumber(): ?string { return $this->vatNumber; }

	public function setVatNumber(?string $vatNumber): self
	{
		$this->vatNumber = $vatNumber;

		return $this;
	}

	public function getCountry(): ?Country { return $this->country; }

	public function setCountry(Country $country): self
	{
		$this->country = $country;

		return $this;
	}

	public function toArray(): array
	{
		return [
			'firstName' => $this->getFirstName(),
			'lastName'  => $this->getLastName(),
			'company'   => $this->company,
			'email'     => $this->email,
			'phone'     => $this->phone,
			'street'    => $this->getStreet(),
			'city'      => $this->getCity(),
			'postal'    => $this->getPostal(),
			'country'   => $this->country ? $this->country->getId() : null,
			'idNumber'  => $this->idNumber,
			'vatNumber' => $this->vatNumber,
		];
	}

	public function fillFromOrderAddress(OrderAddress $address): self
	{
		$this->setFirstName($address->getFirstName());
		$this->setLastName($address->getLastName());
		$this->email       = $address->getEmail();
		$this->phone       = $address->getPhone();
		$this->street      = $address->getStreet();
		$this->city        = $address->getCity();
		$this->postal      = $address->getPostal();
		$this->country     = $address->getCountry();
		$this->idVatNumber = $address->idVatNumber;
		$this->setCompany($address->getCompany());

		if (EshopOrdersConfig::load('customer.disableModifyIco')) {
			if (!$this->idNumber) {
				$this->idNumber = $address->getIdNumber();
			}

			if (!$this->vatNumber) {
				$this->vatNumber = $address->getVatNumber();
			}
		} else {
			$this->idNumber  = $address->getIdNumber();
			$this->vatNumber = $address->getVatNumber();
		}

		return $this;
	}

	public function getDetailString(): string
	{
		$str = '';

		if ($this->addressName) {
			$str .= $this->addressName . ' - ';
		}

		return $str . trim(implode(', ', [$this->getName(), $this->getStreet(), $this->getPostal(), $this->getCity()]), ' ,');
	}

	public function clearFields(): void
	{
		$this->firstName   = null;
		$this->lastName    = null;
		$this->email       = null;
		$this->phone       = null;
		$this->company     = null;
		$this->street      = null;
		$this->city        = null;
		$this->postal      = null;
		$this->idNumber    = null;
		$this->vatNumber   = null;
		$this->idVatNumber = null;

		if ($this->country) {
			$this->country = null;
		}
	}
}
