<?php declare(strict_types = 1);

namespace EshopOrders\Model\Entities;

use Brick\Money\Money;
use Core\Model\Entities\Site;
use Core\Model\Entities\TId;
use Core\Model\Helpers\Arrays;
use Core\Model\Helpers\CoreHelper;
use Core\Model\Parameters;
use Core\Model\Templating\Filters\Price as PriceFilter;
use Currency\DI\CurrencyExtension;
use Currency\Model\Config as CurrencyConfig;
use DateTimeInterface;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\Mapping as ORM;
use EshopOrders\Model\EshopOrdersConfig;
use EshopOrders\Model\Helpers\OrderHelper;
use EshopOrders\Model\PriceCalculator\PriceCalculator;
use EshopOrders\Model\PriceCalculator\PriceCalculatorDiscount;
use EshopOrders\Model\PriceCalculator\PriceCalculatorItem;
use FilesystemIterator;
use Nette\Utils\DateTime;
use Nette\Utils\Random;
use Nette\Utils\Validators;
use RecursiveDirectoryIterator;
use RecursiveIteratorIterator;
use RegexIterator;

#[ORM\Table('eshop_orders__order')]
#[ORM\Entity]
#[ORM\EntityListeners([OrderListener::class])]
class Order
{
	public const paramInternalNote = 'internalNote';

	use TId;

	#[ORM\Column(name: 'ident', type: Types::STRING, length: 255, nullable: false)]
	protected string $ident = '';

	#[ORM\JoinColumn(name: 'customer_id', referencedColumnName: 'id', nullable: true, onDelete: 'SET NULL')]
	#[ORM\ManyToOne(targetEntity: Customer::class, cascade: ['persist'])]
	protected ?Customer $customer = null;

	#[ORM\Column(name: 'message', type: Types::TEXT, nullable: true)]
	protected ?string $message = null;

	#[ORM\JoinColumn(name: 'payment', referencedColumnName: 'id', onDelete: 'SET NULL')]
	#[ORM\OneToOne(targetEntity: OrderPayment::class, cascade: ['persist'])]
	protected ?OrderPayment $payment = null;

	#[ORM\JoinColumn(name: 'spedition', referencedColumnName: 'id', onDelete: 'SET NULL')]
	#[ORM\OneToOne(targetEntity: OrderSpedition::class, cascade: ['persist'])]
	protected ?OrderSpedition $spedition = null;

	/** @var Collection<OrderItem> */
	#[ORM\OneToMany(targetEntity: OrderItem::class, mappedBy: 'order', indexBy: 'id')]
	protected Collection $orderItems;

	/** @var Collection<OrderDiscount> */
	#[ORM\OneToMany(targetEntity: OrderDiscount::class, mappedBy: 'order', indexBy: 'id')]
	public Collection $orderDiscounts;

	#[ORM\JoinColumn(name: 'address_delivery_id', referencedColumnName: 'id', onDelete: 'CASCADE')]
	#[ORM\OneToOne(targetEntity: OrderAddress::class)]
	protected ?OrderAddress $addressDelivery = null;

	#[ORM\JoinColumn(name: 'address_invoice_id', referencedColumnName: 'id', onDelete: 'CASCADE')]
	#[ORM\OneToOne(targetEntity: OrderAddress::class)]
	protected ?OrderAddress $addressInvoice = null;

	/** @var Collection<OrderStatus> */
	#[ORM\OneToMany(targetEntity: OrderStatus::class, mappedBy: 'order', cascade: ['persist'])]
	protected Collection $orderStatuses;

	/** @var Collection<OrderPaidHistory> */
	#[ORM\OneToMany(targetEntity: OrderPaidHistory::class, mappedBy: 'order', cascade: ['persist'])]
	protected Collection $orderPaidHistory;

	#[ORM\Column(name: 'agreed_terms', type: Types::SMALLINT, nullable: true, options: ['default' => 0])]
	protected int $agreedTerms = 0;

	/** @var Collection<OrderFlag> */
	#[ORM\OneToMany(targetEntity: OrderFlag::class, mappedBy: 'order', indexBy: 'type')]
	protected Collection $orderFlags;

	#[ORM\JoinColumn(name: 'invoice_id', referencedColumnName: 'id', nullable: true, onDelete: 'SET NULL')]
	#[ORM\OneToOne(targetEntity: Invoice::class, inversedBy: 'order')]
	protected ?Invoice $invoice = null;

	#[ORM\JoinColumn(name: 'site_id', referencedColumnName: 'ident', nullable: true, onDelete: 'SET NULL')]
	#[ORM\ManyToOne(targetEntity: Site::class)]
	public ?Site $site = null;

	#[ORM\OneToOne(targetEntity: OrderCurrency::class, mappedBy: 'order', cascade: ['persist'])]
	public ?OrderCurrency $currency = null;

	/** @var Collection<OrderGift> */
	#[ORM\OneToMany(targetEntity: OrderGift::class, mappedBy: 'order')]
	protected Collection $gifts;

	#[ORM\Column(name: 'is_paid', type: Types::SMALLINT, options: ['default' => 0])]
	public int $isPaid = 0;

	#[ORM\Column(type: Types::DATETIME_MUTABLE, nullable: true)]
	public ?DateTimeInterface $paid = null;

	#[ORM\JoinColumn(referencedColumnName: 'id', nullable: true, onDelete: 'SET NULL')]
	#[ORM\ManyToOne(targetEntity: Order::class, inversedBy: 'orderForCorrectiveTaxDocument')]
	public ?Order $correctiveTaxDocumentOf = null;

	/** @var Collection<self> */
	#[ORM\OneToMany(targetEntity: Order::class, mappedBy: 'correctiveTaxDocumentOf')]
	public Collection $orderForCorrectiveTaxDocument;

	#[ORM\Column(type: Types::SMALLINT, options: ['default' => 0])]
	public int $isCorrectiveTaxDocument = 0;

	#[ORM\Column(name: 'lang', type: Types::STRING, nullable: false, options: ['default' => 'cs'])]
	public string $lang;

	#[ORM\Column(type: Types::STRING, nullable: true, options: ['default' => null])]
	public ?string $receiptIdent = null;

	#[ORM\Column(type: Types::SMALLINT, options: ['default' => 1])]
	public int $enableInvoiceGeneration = 1;

	public ?PriceFilter $priceFilter = null;

	#[ORM\Column(type: Types::JSON, nullable: true)]
	protected ?array $params = null;

	#[ORM\Column(type: Types::SMALLINT, options: ['default' => 0])]
	public int $isMessageReadyToDelivery = 0;

	#[ORM\Column(type: Types::DATETIME_MUTABLE, nullable: true)]
	public ?\DateTime $demandedExpeditionDate = null;

	public ?PriceCalculator $priceCalculator         = null;
	public ?PriceCalculator $priceCalculatorCurrency = null;

	#[ORM\Column(type: Types::INTEGER, options: ['default' => 0])]
	public int $invoiceReminderCount = 0;

	#[ORM\Column(type: Types::STRING, length: 255, nullable: true, options: ['default' => null])]
	public ?string $ip = null;

	#[ORM\Column(name: '`type`', type: Types::STRING, length: 255, nullable: true, options: ['default' => null])]
	public ?string $type = null;

	public function __construct(Site $site)
	{
		$this->ident                         = $this->genUuid();
		$this->site                          = $site;
		$this->orderItems                    = new ArrayCollection();
		$this->orderDiscounts                = new ArrayCollection();
		$this->orderFlags                    = new ArrayCollection();
		$this->gifts                         = new ArrayCollection();
		$this->orderStatuses                 = new ArrayCollection();
		$this->orderPaidHistory              = new ArrayCollection();
		$this->addressDelivery               = null;
		$this->addressInvoice                = null;
		$this->orderForCorrectiveTaxDocument = new ArrayCollection();
		$this->lang                          = Parameters::loadString('translation.default') ?: 'cs';

		$this->ip = CoreHelper::getUserIpAddress();
	}

	public function initPriceCalculator(): void
	{
		if ($this->priceCalculator === null) {
			$currencyDefault               = $this->getDefaultCurrency();
			$currency                      = $this->getCurrencyCode();
			$this->priceCalculator         = new PriceCalculator($currencyDefault, (bool) $this->isCorrectiveTaxDocument);
			$this->priceCalculatorCurrency = new PriceCalculator($currency, (bool) $this->isCorrectiveTaxDocument);

			$isReceipt = in_array($this->getPaymentIdent(), ['storeCash', 'storeCard']);

			foreach ($this->getItemsRaw() as $item) {
				$calculatorItem = new PriceCalculatorItem(
					'',
					$item->getPriceRaw(),
					$item->getQuantity(),
					$item->getVatRate(),
					$currencyDefault,
					$item->getMoreDataValue('discountDisabled') && (bool) $item->getMoreDataValue('discountDisabled')
				);

				$calculatorItemCurrency = new PriceCalculatorItem(
					'',
					$this->calculateCurrencyPrice($item->getPriceRaw()),
					$item->getQuantity(),
					$item->getVatRate(),
					$currency,
					$item->getMoreDataValue('discountDisabled') && (bool) $item->getMoreDataValue('discountDisabled')
				);

				$item->priceCalculatorItem         = $calculatorItem;
				$item->priceCalculatorItemCurrency = $calculatorItemCurrency;

				$key = (string) $item->getId();
				if (!$key) {
					$key = Random::generate();
				}

				$this->priceCalculator->addItem($key, $calculatorItem);
				$this->priceCalculatorCurrency->addItem((string) $item->getId(), $calculatorItemCurrency);

				if ($isReceipt) {
					foreach ($item->sales as $sale) {
						$salePriceItem         = new PriceCalculatorDiscount(
							'fix',
							(float) $sale->getSaleValue(),
							$currencyDefault,
						);
						$salePriceItemCurrency = new PriceCalculatorDiscount(
							'fix',
							$this->calculateCurrencyPrice((float) $sale->getSaleValue()),
							$currency,
						);

						$sale->priceCalculatorItem         = $salePriceItem;
						$sale->priceCalculatorItemCurrency = $salePriceItemCurrency;

						$this->priceCalculator->addDiscountCode(Random::generate(), $salePriceItem);
						$this->priceCalculatorCurrency->addDiscountCode(Random::generate(), $salePriceItemCurrency);
					}
				}
			}

			foreach ($this->orderDiscounts as $discount) {
				$calculatorItem = new PriceCalculatorDiscount(
					$discount->getType(),
					$discount->getPriceRaw(),
					$currencyDefault
				);

				$calculatorItemCurrency = new PriceCalculatorDiscount(
					$discount->getType(),
					$this->calculateCurrencyPrice($discount->getPriceRaw()),
					$currency
				);

				$discount->priceCalculatorItem         = $calculatorItem;
				$discount->priceCalculatorItemCurrency = $calculatorItemCurrency;

				$this->priceCalculator->addDiscountCode((string) $discount->getId(), $calculatorItem);
				$this->priceCalculatorCurrency->addDiscountCode((string) $discount->getId(), $calculatorItemCurrency);
			}

			$spedition = $this->getSpedition();
			if ($spedition) {
				$calculatorItem = new PriceCalculatorItem(
					$spedition->getName(),
					$spedition->getPriceRaw(),
					1,
					$spedition->getVatRate(),
					$currencyDefault,
					false
				);

				$calculatorItemCurrency = new PriceCalculatorItem(
					$spedition->getName(),
					$this->calculateCurrencyPrice($spedition->getPriceRaw()),
					1,
					$spedition->getVatRate(),
					$currency,
					false
				);

				$spedition->priceCalculatorItem         = $calculatorItem;
				$spedition->priceCalculatorItemCurrency = $calculatorItemCurrency;

				$this->priceCalculator->setSpedition($calculatorItem);
				$this->priceCalculatorCurrency->setSpedition($calculatorItemCurrency);
			}

			$payment = $this->getPayment();
			if ($payment) {
				$calculatorItem = new PriceCalculatorItem(
					$payment->getName(),
					$payment->getPriceRaw(),
					1,
					$payment->getVatRate(),
					$currencyDefault,
					false
				);

				$calculatorItemCurrency = new PriceCalculatorItem(
					$payment->getName(),
					$this->calculateCurrencyPrice($payment->getPriceRaw()),
					1,
					$payment->getVatRate(),
					$currency,
					false
				);

				$payment->priceCalculatorItem         = $calculatorItem;
				$payment->priceCalculatorItemCurrency = $calculatorItemCurrency;

				$this->priceCalculator->setPayment($calculatorItem);
				$this->priceCalculatorCurrency->setPayment($calculatorItemCurrency);
			}
		}
	}

	public function reloadPriceCalculator(): void
	{
		$this->priceCalculator         = null;
		$this->priceCalculatorCurrency = null;
	}

	public function getIdent(): string { return $this->ident; }

	public function getOrderForCorrectiveTaxDocument(): ?self
	{
		if ($this->orderForCorrectiveTaxDocument instanceof self) {
			return $this->orderForCorrectiveTaxDocument;
		}

		$first = $this->orderForCorrectiveTaxDocument->first();
		if ($first !== false) {
			return $first;
		}

		return null;
	}

	public function getCustomer(): ?Customer { return $this->customer; }

	public function getMessage(): ?string { return $this->message; }

	public function getPrice(bool $useCurrency = false): float
	{
		$this->initPriceCalculator();

		if ($this->isZeroVat()) {
			return $this->getPriceWithoutVat($useCurrency);
		}

		return $useCurrency
			? $this->priceCalculatorCurrency->getWithVatTotal()->getAmount()->toFloat()
			: $this->priceCalculator->getWithVatTotal()->getAmount()->toFloat();
	}

	public function getPriceWithoutVat(bool $useCurrency = false): float
	{
		$this->initPriceCalculator();

		return $useCurrency
			? $this->priceCalculatorCurrency->getWithoutVatTotal()->getAmount()->toFloat()
			: $this->priceCalculator->getWithoutVatTotal()->getAmount()->toFloat();
	}

	public function getPriceItems(bool $useCurrency = false): float
	{
		$this->initPriceCalculator();

		if ($this->isZeroVat()) {
			return $this->getItemsPriceWithoutVat($useCurrency);
		}

		return $useCurrency
			? $this->priceCalculatorCurrency->getWithVatItemsWithoutDiscount()->getAmount()->toFloat()
			: $this->priceCalculator->getWithVatItemsWithoutDiscount()->getAmount()->toFloat();
	}

	public function getItemsPriceWithoutVat(bool $useCurrency = false): float
	{
		$this->initPriceCalculator();

		return $useCurrency
			? $this->priceCalculatorCurrency->getWithoutVatItemsWithoutDiscount()->getAmount()->toFloat()
			: $this->priceCalculator->getWithoutVatItemsWithoutDiscount()->getAmount()->toFloat();
	}

	public function getPaySpedPrice(bool $useCurrency = false): float
	{
		$this->initPriceCalculator();

		if ($this->isZeroVat()) {
			return $this->getPaySpedPriceWithoutVat($useCurrency);
		}

		$priceTotal = 0;
		$priceTotal += $this->getSpedition() ? $this->getSpedition()->getPrice($useCurrency) : 0;

		return $priceTotal + ($this->getPayment() ? $this->getPayment()->getPrice($useCurrency) : 0);
	}

	public function getPaySpedPriceWithoutVat(bool $useCurrency = false): float
	{
		$this->initPriceCalculator();

		$priceTotal = 0;
		$priceTotal += $this->getSpedition() ? $this->getSpedition()->getPriceWithoutVat($useCurrency) : 0;

		return $priceTotal + ($this->getPayment() ? $this->getPayment()->getPriceWithoutVat($useCurrency) : 0);
	}

	public function getPaySpedVatRate(): int
	{
		$this->initPriceCalculator();

		if ($this->isZeroVat()) {
			return 0;
		}

		if ($this->getSpedition()) {
			return $this->getSpedition()->getVatRate();
		}

		return (int) EshopOrdersConfig::load('speditions.vat');
	}

	public function getVatRates(bool $useCurrency = false): array
	{
		$this->initPriceCalculator();

		return array_map(static function($vals) {
			return [
				'withoutVat' => $vals['withoutVat']->getAmount()->toFloat(),
				'total'      => $vals['total']->getAmount()->toFloat(),
				'rate'       => $vals['rate']->getAmount()->toFloat(),
			];
		}, $useCurrency
			? $this->priceCalculatorCurrency->getVatBreakdown()
			: $this->priceCalculator->getVatBreakdown());
	}

	public function getDiscountsByVat(bool $useCurrency = false): array
	{
		$this->initPriceCalculator();

		return array_map(static function($vals) {
			return [
				'withoutVat' => $vals['withoutVat']->getAmount()->toFloat(),
				'total'      => $vals['total']->getAmount()->toFloat(),
			];
		}, $useCurrency
			? $this->priceCalculatorCurrency->getDiscountsByVat()
			: $this->priceCalculator->getDiscountsByVat());
	}

	public function getDiscountsCodes(): array { return array_map(static fn(OrderDiscount $d) => $d->getCode() ?: $d->getName(), $this->getOrderDiscounts()->toArray()); }

	/**
	 * @deprecated replace with getPriceItems
	 */
	public function getPriceWithoutDiscounts(bool $useCurrency = false): float
	{
		return $this->getPriceItems($useCurrency);
	}

	public function getPriceItemsDiscount(bool $useCurrency = false): float
	{
		$this->initPriceCalculator();

		if ($this->isZeroVat()) {
			return $this->getPriceItemsWithoutVatDiscount($useCurrency);
		}

		return $useCurrency
			? $this->priceCalculatorCurrency->getWithVatItems()->getAmount()->toFloat()
			: $this->priceCalculator->getWithVatItems()->getAmount()->toFloat();
	}

	public function getPriceItemsWithoutVatDiscount(bool $useCurrency = false): float
	{
		$this->initPriceCalculator();

		return $useCurrency
			? $this->priceCalculatorCurrency->getWithoutVatItems()->getAmount()->toFloat()
			: $this->priceCalculator->getWithoutVatItems()->getAmount()->toFloat();
	}

	public function getItemsVatPrice(bool $useCurrency = false): float
	{
		$this->initPriceCalculator();

		$currency = $useCurrency
			? $this->priceCalculatorCurrency->getCurrency()
			: $this->priceCalculator->getCurrency();

		$result = Money::of(0, $currency);

		foreach ($useCurrency
			         ? $this->priceCalculatorCurrency->getItems()
			         : $this->priceCalculator->getItems() as $item) {
			$result = $result->plus($item->getVat());
		}

		return $result->getAmount()->toFloat();
	}

	public function getPaySpedId(): string { return $this->getSpedition()->getId() . '-' . $this->getPayment()->getId(); }

	public function getPaySpedName(): string { return $this->getSpedition()->getName() . ' - ' . $this->getPayment()->getName(); }

	public function getPayment(): ?OrderPayment
	{
		if ($this->payment && $this->payment->order === null) {
			$this->payment->order = $this;
		}

		return $this->payment;
	}

	public function getPaymentIdent(): ?string
	{
		return $this->getPayment() && $this->getPayment()->getPayment() ? $this->getPayment()->getPayment()->getIdent() : null;
	}

	public function getEshopPaymentId(): ?int { return $this->getPayment() ? $this->getPayment()->getPaymentId() : null; }

	public function getSpedition(): ?OrderSpedition
	{
		if ($this->spedition && $this->spedition->order === null) {
			$this->spedition->order = $this;
		}

		return $this->spedition;
	}

	public function getSpeditionIdent(): ?string { return $this->getSpedition() ? $this->getSpedition()->getIdent() : null; }

	public function getEshopSpeditionId(): ?int { return $this->getSpedition() ? $this->getSpedition()->getSpeditionId() : null; }

	/** @return Collection<OrderItem> */
	public function getItemsRaw(): Collection
	{
		return $this->orderItems;
	}

	/**
	 * @return ArrayCollection<OrderItem>
	 */
	public function getOrderItems(): Collection
	{
		$items   = [];
		$parents = [];

		foreach ($this->orderItems as $k => $item) {
			if ($item->getId()) {
				if ($item->getParent()) {
					$parents[$item->getParent()->getId()][$item->getId()] = $item;
				} else {
					$items[$item->getId()] = $item;
				}
			} else {
				$items[$k] = $item;
			}
		}

		foreach ($parents as $k => $list) {
			Arrays::insertAfter($items, $k, $list);
		}

		$parents = null;

		return new ArrayCollection($items);
	}

	public function getOrderItemByProductId(int $productId): ?OrderItem
	{
		foreach ($this->getOrderItems() as $item) {
			if ($item->getProductId() === $productId) {
				return $item;
			}
		}

		return null;
	}

	public function getAddressDelivery(): ?OrderAddress { return $this->addressDelivery ?: $this->addressInvoice; }

	public function getAddressDeliveryRaw(): ?OrderAddress { return $this->addressDelivery; }

	public function getAddressInvoice(): ?OrderAddress { return $this->addressInvoice ?: ($this->addressDelivery ?: null); }

	public function getAddressInvoiceRaw(): ?OrderAddress { return $this->addressInvoice; }

	public function setCustomer(Customer $customer): Order
	{
		$this->customer = $customer;

		return $this;
	}

	public function setMessage(?string $message): self
	{
		$this->message = $message;

		return $this;
	}

	public function setPayment(OrderPayment $payment): self
	{
		$this->payment = $payment;

		$this->reloadPriceCalculator();

		return $this;
	}

	public function setSpedition(OrderSpedition $spedition): self
	{
		$this->spedition = $spedition;

		$this->reloadPriceCalculator();

		return $this;
	}

	/** @param OrderItem[] $orderItems */
	public function setOrderItems(array $orderItems): self
	{
		$this->orderItems = new ArrayCollection($orderItems);

		if ($this->priceCalculator) {
			$this->reloadPriceCalculator();
		}

		return $this;
	}

	public function setAddressDelivery(OrderAddress $addressDelivery): self
	{
		$this->addressDelivery = $addressDelivery;

		return $this;
	}

	public function setAddressInvoice(OrderAddress $addressInvoice): self
	{
		$this->addressInvoice = $addressInvoice;

		return $this;
	}

	/** @return Collection<OrderDiscount> */
	public function getOrderDiscounts(): Collection { return $this->orderDiscounts; }

	/** @param OrderDiscount[] $orderDiscounts */
	public function setOrderDiscounts(array $orderDiscounts): Order
	{
		$this->orderDiscounts = new ArrayCollection($orderDiscounts);

		return $this;
	}

	public function addOrderDiscount(OrderDiscount $orderDiscount): self
	{
		$this->orderDiscounts->add($orderDiscount);

		return $this;
	}

	/**
	 * TODO remove
	 * @deprecated
	 */
	protected function calculateDiscounts(float $price): float
	{
		foreach ($this->getOrderDiscounts() as $discount) {
			$price += $discount->calculateDiscount($price);
		}

		return $price;
	}

	public function getAgreedTerms(): bool { return (bool) $this->agreedTerms; }

	/** @param bool|int $agreedTerms */
	public function setAgreedTerms($agreedTerms): self
	{
		$this->agreedTerms = (int) $agreedTerms;

		return $this;
	}

	/** @return Collection<OrderFlag> */
	public function getOrderFlags(): Collection { return $this->orderFlags; }

	/** @param OrderFlag[] $orderFlags */
	public function setOrderFlags(array $orderFlags): self
	{
		$this->orderFlags = new ArrayCollection($orderFlags);

		return $this;
	}

	public function addFlag(OrderFlag $flag): self
	{
		if (!$this->orderFlags->containsKey($flag->getType())) {
			$this->orderFlags->set($flag->getType(), $flag);
		}

		return $this;
	}

	public function hasFlag(string $type): bool
	{
		return ($this->orderFlags->containsKey($type) && $this->orderFlags[$type] == true);
	}

	/** @return Collection<OrderStatus> */
	public function getOrderStatuses(): Collection { return $this->orderStatuses; }

	public function getNewestOrderStatus(): ?OrderStatus
	{
		$last = null;

		foreach ($this->orderStatuses as $status) {
			if (
				!$status->isDeleted()
				&& (!$last || $status->getCreated() > $last->getCreated())
			) {
				$last = $status;
			}
		}

		return $last ?: $this->getOrderStatuses()->last();
	}

	/** @param OrderStatus[] $orderStatuses */
	public function setOrderStatuses(array $orderStatuses): Order
	{
		$this->orderStatuses = new ArrayCollection($orderStatuses);

		return $this;
	}

	public function getCreatedTime(): ?\DateTime
	{
		foreach ($this->orderStatuses as $orderStatus) {
			if ($orderStatus->getStatus()->getId() === 'created') {
				$createdStatus = $orderStatus;
				break;
			}
		}

		if (isset($createdStatus)) {
			return $createdStatus->getCreated();
		}

		return null;
	}

	public function getDUZP(): ?DateTimeInterface
	{
		if ($this->invoice) {
			return $this->invoice->getDUZP();
		}

		$now = new DateTime;

		if (EshopOrdersConfig::load('invoice.document.separatedDUZPCreated')) {
			return $this->getCreatedTime() ?: $now;
		}

		return $now;
	}

	public function getInvoice(): ?Invoice { return $this->invoice; }

	public function setInvoice(?Invoice $invoice): void { $this->invoice = $invoice; }

	public function getCurrencyCode(): string
	{
		if (class_exists(CurrencyExtension::class) && $this->currency) {
			return $this->currency->code;
		}

		return $this->getDefaultCurrency();
	}

	public function getDefaultCurrency(): string { return class_exists(CurrencyConfig::class) ? CurrencyConfig::loadString('default') : 'CZK'; }

	public function calculateCurrencyPrice(float $price): float
	{
		return $this->currency
			? OrderHelper::calculateCurrencyPrice($price, (float) $this->currency->rate, (int) $this->currency->decimals)
			: $price;
	}

	public ?int $renderPriceDecimals = null;

	public function renderPrice(float $price, bool $useCurrency = false, ?string $method = null, ?bool $currencyFromLeft = null, ?string $separator = null, ?string $decSep = null, ?int $decimals = null): string
	{
		if (!$method) {
			$method = 'format';
		}

		if (!$decimals && $this->renderPriceDecimals) {
			$decimals = $this->renderPriceDecimals;
		}

		return $this->priceFilter->$method($price, $useCurrency ? $this->getCurrencyCode() : $this->getDefaultCurrency(), $currencyFromLeft, $separator, $decSep, $decimals);
	}

	/*************************************
	 * == Gifts
	 */

	/** @return Collection<OrderGift> */
	public function getGifts(): Collection { return $this->gifts; }

	public function addGift(OrderGift $gift): self
	{
		$this->gifts->add($gift);

		return $this;
	}

	/** http://stackoverflow.com/a/2040279
	 * @return string UUID - unikatni ID, tezke na uhodnuti
	 */
	protected function genUuid(): string
	{
		return sprintf('%04x%04x-%04x-%04x-%04x-%04x%04x%04x',
			// 32 bits for "time_low"
			mt_rand(0, 0xffff), mt_rand(0, 0xffff),

			// 16 bits for "time_mid"
			mt_rand(0, 0xffff),

			// 16 bits for "time_hi_and_version",
			// four most significant bits holds version number 4
			mt_rand(0, 0x0fff) | 0x4000,

			// 16 bits, 8 bits for "clk_seq_hi_res",
			// 8 bits for "clk_seq_low",
			// two most significant bits holds zero and one for variant DCE1.1
			mt_rand(0, 0x3fff) | 0x8000,

			// 48 bits for "node"
			mt_rand(0, 0xffff), mt_rand(0, 0xffff), mt_rand(0, 0xffff)
		);
	}

	/** @return OrderItem[] */
	public function getItemsForDiscount(): array
	{
		$items = [];

		foreach ($this->orderItems as $k => $item) {
			if ($item->getMoreDataValue('discountDisabled', false)) {
				continue;
			}

			$items[$k] = $item;
		}

		return $items;
	}

	public function isZeroVat(): bool
	{
		$addrInv = $this->getAddressInvoice();

		if (!$addrInv || !$addrInv->getCountry()) {
			return false;
		}

		$vatRate = OrderHelper::checkCountryVatRate(
			21,
			$addrInv->getCountry()->getId(),
			(bool) $addrInv->validatedVatNumber,
			$addrInv->getIdNumber(),
			$addrInv->getVatNumber(),
		);

		return $vatRate === 0;
	}

	public function getPaidDate(): ?\DateTimeInterface { return $this->isPaid ? $this->paid : null; }

	public function __clone()
	{
		$this->id    = null;
		$this->ident = $this->genUuid();
	}

	public function isCanceled(): bool
	{
		$isCanceled = false;
		/** @var OrderStatus $orderStatus */
		foreach ($this->getOrderStatuses()->toArray() as $orderStatus) {
			if ($orderStatus->getStatus()->getId() === OrderStatus::STATUS_CANCELED && !$orderStatus->isDeleted()) {
				$isCanceled = true;
			}
		}

		return $isCanceled;
	}

	public function isShipped(): bool
	{
		$isShipped = false;
		/** @var OrderStatus $orderStatus */
		foreach ($this->getOrderStatuses()->toArray() as $orderStatus) {
			if ($orderStatus->getStatus()->getId() === OrderStatus::STATUS_SPEDITION && !$orderStatus->isDeleted()) {
				$isShipped = true;
			}
		}

		return $isShipped;
	}

	public function isFinished(): bool
	{
		$isShipped = false;
		/** @var OrderStatus $orderStatus */
		foreach ($this->getOrderStatuses()->toArray() as $orderStatus) {
			if ($orderStatus->getStatus()->getId() === OrderStatus::STATUS_FINISHED && !$orderStatus->isDeleted()) {
				$isShipped = true;
			}
		}

		return $isShipped;
	}

	public function getItemsWeight(bool $inKg = true): float
	{
		$weight = 0;

		foreach ($this->getOrderItems() as $orderItem) {
			if ($orderItem->getProduct() && $orderItem->getProduct()->weight) {
				$weight += $orderItem->getProduct()->weight;
			}
		}

		return round($inKg ? $weight / 1000 : $weight, 2);
	}

	public function hasPreorderProduct(): bool
	{
		foreach ($this->getOrderItems() as $item) {
			if ($item->getMoreDataValue('isPreorder', false)) {
				return true;
			}
		}

		return false;
	}

	public function isOtherCountryVatFilled(): bool
	{
		$addrInv = $this->getAddressInvoice();
		$country = $addrInv && $addrInv->getCountry() ? mb_strtoupper($addrInv->getCountry()->getId()) : null;

		return OrderHelper::isOtherCountryVatFilled($country, $addrInv ? $addrInv->getVatNumber() : null);
	}

	public function isOrderItemsReadyToDelivery(): bool
	{
		foreach ($this->orderItems as $orderItem) {
			if (!$orderItem->isReadyToDelivery) {
				return false;
			}
		}

		foreach ($this->getGifts() as $gift) {
			if (!$gift->isReadyToDelivery) {
				return false;
			}
		}

		return true;
	}

	public function getParams(): array { return $this->params ?: []; }

	/** @return mixed */
	public function getParam(string $key) { return $this->getParams()[$key] ?? null; }

	/** @param mixed $value */
	public function setParam(string $key, $value): void
	{
		if (!is_array($this->params)) {
			$this->params = [];
		}

		$this->params[$key] = $value;
	}

	public function removeParam(string $key): void
	{
		if (is_array($this->params)) {
			unset($this->params[$key]);
		}
	}

	public function isMessageReadyToDelivery(): bool
	{
		return Validators::isNone($this->getMessage()) || $this->isMessageReadyToDelivery;
	}

	public function increaseInvoiceReminderCount(): void
	{
		/** @phpstan-ignore-next-line */
		$max                        = (int) max(Customer::$invoiceReminderOptions);
		$this->invoiceReminderCount = min($this->invoiceReminderCount + 1, $max);

		if ($this->getCustomer()) {
			$this->getCustomer()->invoiceReminderCount = $this->invoiceReminderCount;
		}
	}

	public function getAttachedFiles(): array
	{
		$result   = [];
		$basePath = OrderHelper::getOrderAttachmentDir($this->getId());

		if (!is_dir($basePath)) {
			return $result;
		}

		$iterator = new RecursiveIteratorIterator(
			new RecursiveDirectoryIterator($basePath, FilesystemIterator::SKIP_DOTS)
		);

		foreach (array_keys(iterator_to_array(new RegexIterator($iterator, "/.*$/i"))) as $file) {
			$file = str_replace($basePath . DIRECTORY_SEPARATOR, '', $file);
			$file = str_replace('\\', '/', $file);

			$result[] = $file;
		}

		return $result;
	}

	public function isCheckoutOrder(): bool
	{
		return in_array($this->getPaymentIdent(), ['storeCash', 'storeCard']);
	}

}
