<?php declare(strict_types = 1);

namespace EshopOrders\Model\Entities;

use Core\Model\Entities\TId;
use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\Mapping as ORM;
use EshopOrders\Model\Helpers\OrderHelper;
use EshopOrders\Model\PriceCalculator\PriceCalculatorDiscount;

#[ORM\Table('eshop_orders__order_item_sale')]
#[ORM\Entity]
class OrderItemSale
{
	use TId;

	public const TYPE_PERCENT_ONE = 'percentOne';
	public const TYPE_PERCENT_ALL = 'percentAll';
	public const TYPE_FIX_ONE     = 'fixOne';
	public const TYPE_FIX_ALL     = 'fixAll';

	public ?PriceCalculatorDiscount $priceCalculatorItem         = null;
	public ?PriceCalculatorDiscount $priceCalculatorItemCurrency = null;

	#[ORM\Column(name: 'name', type: Types::STRING, nullable: true)]
	protected ?string $name = null;

	#[ORM\Column(name: 'description', type: Types::STRING, length: 1000, nullable: true)]
	protected ?string $description = null;

	/** sleva v procentech nebo pevna castka
	 * @var float
	 */
	#[ORM\Column(name: 'value', type: Types::DECIMAL, precision: 10, scale: 2, nullable: false, options: ['default' => 0])]
	protected $value;

	#[ORM\Column(name: 'type', type: Types::STRING, nullable: false)]
	protected string $type;

	#[ORM\JoinColumn(name: 'order_item_id', referencedColumnName: 'id', onDelete: 'CASCADE')]
	#[ORM\ManyToOne(targetEntity: OrderItem::class, inversedBy: 'sales')]
	public OrderItem $orderItem;

	/**
	 * @param float $value
	 */
	public function __construct($value, OrderItem $orderItem, string $type)
	{
		$this->value     = $value;
		$this->orderItem = $orderItem;
		$this->type      = $type;
	}

	public function getName(): ?string { return $this->name; }

	public function setName(string $name): OrderItemSale
	{
		$this->name = $name;

		return $this;
	}

	public function getType(): string { return $this->type; }

	public function setType(string $type): self
	{
		$this->type = $type;

		return $this;
	}

	public function getDescription(): ?string { return $this->description; }

	public function setDescription(string $description): OrderItemSale
	{
		$this->description = $description;

		return $this;
	}

	public function getValue(): float
	{
		return (float) $this->value;
	}

	public function setValue(float $value): OrderItemSale
	{
		$this->value = $value;

		return $this;
	}

	public function getOrderItem(): OrderItem { return $this->orderItem; }

	public function setOrderItem(OrderItem $orderItem): OrderItemSale
	{
		$this->orderItem = $orderItem;

		return $this;
	}

	/** @return float|int|mixed */
	public function getSaleValue(bool $useCurrency = false)
	{
		$basePrice = $this->orderItem->getPrice($useCurrency);
		$quantity  = $this->orderItem->getQuantity();

		return OrderHelper::calculateSaleValue($this->type, $this->getValue(), $quantity, $basePrice);
	}

	public function getVatRate(): int { return $this->orderItem->getVatRate(); }

	public function getFormatted(): string
	{
		$price = $this->orderItem->getOrder()->renderPrice($this->getValue());
		$this->orderItem->getQuantity();

		switch ($this->type) {
			case self::TYPE_FIX_ALL:
				return '-' . $price . ' na produkt';
			case self::TYPE_FIX_ONE:
				return '-' . $price . ' na ks';
			case self::TYPE_PERCENT_ALL:
				return '-' . $this->value . '% na produkt';
			case self::TYPE_PERCENT_ONE:
				return '-' . $this->value . '% na ks';
		}

		return '';
	}
}
