<?php declare(strict_types = 1);

namespace EshopOrders\Model\Entities;

use Contributte\Translation\Exceptions\InvalidArgument;
use Contributte\Translation\Translator;
use Core\Model\Application\AppState;
use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Event\EventDispatcher;
use Core\Model\Templating\Filters\Price as PriceFilter;
use Doctrine;
use Doctrine\ORM\Mapping as ORM;
use Doctrine\Persistence\Event\LifecycleEventArgs;
use EshopOrders\FrontModule\Model\Event\OrderEvent;
use EshopOrders\Model\Event\OrderStatusEvent;
use Exception;
use Nette\SmartObject;
use Nette\Utils\DateTime;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Tracy\Debugger;

class OrderListener implements EventSubscriberInterface
{
	use SmartObject;

	public function __construct(
		protected PriceFilter            $priceFilter,
		protected EventDispatcher        $eventDispatcher,
		protected Translator             $translator,
		protected EntityManagerDecorator $em,
	)
	{
	}

	public static function getSubscribedEvents(): array
	{
		return [];
	}

	#[ORM\PrePersist]
	public function prePersistHandler(Order $order, LifecycleEventArgs $event): void
	{
		$em = $event->getObjectManager();
		if ($order->isPaid === 1 && $order->getId() === null) {
			$em->persist($this->createOrderPaidHistory($order, true));
		}
	}

	#[ORM\PostUpdate]
	public function postUpdateHandler(Order $order, LifecycleEventArgs $event): void
	{
		$em        = $this->em;
		$changeSet = $em->getUnitOfWork()->getEntityChangeSet($order);

		try {
			if ($changeSet && isset($changeSet['isPaid'])) {
				$isPaid           = $changeSet['isPaid'][1] === 1;
				$orderPaidHistory = $this->createOrderPaidHistory($order, $isPaid);
				$em->persist($orderPaidHistory);

				if ($isPaid) {
					$order->paid = new DateTime;
					$this->eventDispatcher->dispatch(new OrderStatusEvent($order, OrderStatus::STATUS_IS_PAID), Order::class . '::changeStatus');
				} else {
					$em->flush();
				}
			}
		} catch (Exception $e) {
			Debugger::log($e, 'order-listener-log-paid');
		}
	}

	#[ORM\PostLoad]
	public function postLoad(Order $order, LifecycleEventArgs $args): void
	{
		$order->priceFilter = clone $this->priceFilter;

		$this->eventDispatcher->dispatch(new OrderEvent($order), Order::class . '::postLoad');
	}

	/**
	 *
	 * @throws InvalidArgument
	 */
	protected function createOrderPaidHistory(Order $order, bool $isPaid): OrderPaidHistory
	{
		$eshopOrdersPaymentMethodText = $this->translator->translate(
			AppState::getState('eshopOrdersPaymentMethod', 'eshopOrders.paymentMethod.default')
		);
		$orderPaidHistory             = new OrderPaidHistory($order, $eshopOrdersPaymentMethodText);
		$orderPaidHistory->isDeleted  = (int) !$isPaid;

		return $orderPaidHistory;
	}

}
