<?php declare(strict_types = 1);

namespace EshopOrders\Model\Entities;

use Core\Model\Entities\TId;
use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\Mapping as ORM;
use EshopOrders\Model\Listeners\OrderStatusListener;
use Nette\Utils\DateTime;
use Users\Model\Entities\User;

#[ORM\Table('eshop_orders__order_status')]
#[ORM\Entity]
#[ORM\EntityListeners([OrderStatusListener::class])]
class OrderStatus implements IOrderStatus
{
	use TId;

	public const STATUS_CANCELED            = 'canceled';
	public const STATUS_CREATED             = 'created';
	public const STATUS_FINISHED            = 'finished';
	public const STATUS_PROCESSING          = 'processing';
	public const STATUS_SPEDITION           = 'spedition';
	public const STATUS_WAITING_FOR_PAYMENT = 'waitingForPayment';
	public const STATUS_IS_PAID             = 'isPaid';
	public const STATUS_READY_FOR_PICKUP    = 'readyForPickup';
	public const STATUS_RESERVED            = 'reserved';

	public static array $statuses = [
		self::STATUS_CANCELED            => [],
		self::STATUS_CREATED             => [],
		self::STATUS_FINISHED            => [],
		self::STATUS_PROCESSING          => [],
		self::STATUS_SPEDITION           => [],
		self::STATUS_WAITING_FOR_PAYMENT => [],
		self::STATUS_IS_PAID             => [],
		self::STATUS_READY_FOR_PICKUP    => [],
		self::STATUS_RESERVED            => [],
	];

	#[ORM\JoinColumn(name: 'order_id', referencedColumnName: 'id', onDelete: 'CASCADE')]
	#[ORM\ManyToOne(targetEntity: Order::class, inversedBy: 'orderStatuses')]
	protected Order $order;

	#[ORM\JoinColumn(name: 'status_id', referencedColumnName: 'id', onDelete: 'CASCADE')]
	#[ORM\ManyToOne(targetEntity: Status::class)]
	protected Status $status;

	#[ORM\Column(name: 'created', type: Types::DATETIME_MUTABLE)]
	protected ?\DateTime $created;

	#[ORM\JoinColumn(name: 'created_by', referencedColumnName: 'id', onDelete: 'SET NULL')]
	#[ORM\ManyToOne(targetEntity: User::class)]
	protected ?User $createdBy;

	#[ORM\Column(name: 'parameter', type: Types::STRING, length: 255, nullable: true)]
	protected ?string $parameter = null;

	#[ORM\Column(name: 'message', type: Types::TEXT, nullable: true)]
	protected ?string $message = null;

	#[ORM\Column(name: 'deleted', type: Types::SMALLINT, nullable: true)]
	protected ?int $deleted = null;

	public function __construct(Order $order, Status $status, ?User $createdBy = null)
	{
		$this->order     = $order;
		$this->status    = $status;
		$this->createdBy = $createdBy;
		$this->created   = new DateTime();
	}

	public function getOrder(): Order { return $this->order; }

	public function getStatus(): Status { return $this->status; }

	public function getCreated(): \DateTime { return $this->created; }

	public function getCreatedBy(): User { return $this->createdBy; }

	public function getParameter(): ?string { return $this->parameter; }

	public function setParameter(?string $parameter): OrderStatus
	{
		$this->parameter = $parameter;

		return $this;
	}

	public function getMessage(): ?string { return $this->message; }

	public function setMessage(?string $message): OrderStatus
	{
		$this->message = $message;

		return $this;
	}

	public function setOrder(Order $order): OrderStatus
	{
		$this->order = $order;

		return $this;
	}

	public function isDeleted(): int { return $this->deleted ?: 0; }

	public function setDeleted(bool|int $deleted): OrderStatus
	{
		$this->deleted = (int) $deleted;

		return $this;
	}
}

