<?php declare(strict_types = 1);

namespace EshopOrders\Model\Entities;

use Core\Model\Entities\Country;
use Core\Model\Entities\TId;
use Core\Model\Helpers\Strings;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\Mapping as ORM;
use Gedmo\Mapping\Annotation as Gedmo;

#[ORM\Table('eshop_orders__payment')]
#[ORM\Entity]
class Payment
{
	use TId;

	#[ORM\Column(name: 'name', type: Types::STRING, length: 255, nullable: true)]
	protected ?string $name = null;

	#[ORM\Column(name: 'ident', type: Types::STRING, length: 255, unique: true)]
	protected ?string $ident = null;

	#[ORM\Column(type: Types::STRING, nullable: true)]
	public ?string $code1 = null;

	#[ORM\Column(name: Types::TEXT, type: Types::STRING, length: 65535, nullable: true)]
	protected ?string $text = null;

	#[ORM\Column(name: 'is_published', type: Types::SMALLINT)]
	public ?int $isPublished = 0;

	/** @Gedmo\SortablePosition */
	#[ORM\Column(name: 'position', type: Types::INTEGER)]
	protected ?int $position = 0;

	/**
	 * @var float|string
	 */
	#[ORM\Column(name: 'price', type: Types::DECIMAL, precision: 10, scale: 2)]
	protected $price;

	#[ORM\Column(name: 'currency', type: Types::STRING, nullable: true)]
	public ?string $currency = null;

	#[ORM\Column(name: 'free_from', type: Types::INTEGER, nullable: true)]
	protected ?int $freeFrom = null;

	#[ORM\Column(name: 'available_from', type: Types::INTEGER, nullable: true)]
	protected ?int $availableFrom = null;

	#[ORM\Column(name: 'available_to', type: Types::INTEGER, nullable: true)]
	protected ?int $availableTo = null;

	#[ORM\Column(name: 'image', type: Types::STRING, length: 255, nullable: true)]
	public ?string $image = null;

	/** @var Collection<Customer> */
	#[ORM\ManyToMany(targetEntity: Customer::class, mappedBy: 'disabledPayments', indexBy: 'id')]
	protected Collection $disabledForCustomers;

	#[ORM\Column(name: 'disable_in_google', type: Types::SMALLINT, length: 1, options: ['default' => 0])]
	public int $disableInGoogle = 0;

	#[ORM\Column(name: 'pohoda_name', type: Types::STRING, length: 255, nullable: true)]
	public ?string $pohodaName = null;

	#[ORM\JoinColumn(name: 'country_id', referencedColumnName: 'id', onDelete: 'SET NULL')]
	#[ORM\ManyToOne(targetEntity: Country::class)]
	public ?Country $country = null;

	public function __construct()
	{
		$this->isPublished = 1;
		$this->setPosition(-1);
		$this->freeFrom             = 999999;
		$this->disabledForCustomers = new ArrayCollection();
	}

	public function getName(): ?string { return $this->name; }

	public function getPosition(): int { return $this->position; }

	public function getPrice(): float { return (float) $this->price; }

	public function setName(?string $name): self
	{
		$this->name = $name;

		return $this;
	}

	public function setPosition(int $position): self
	{
		$this->position = $position;

		return $this;
	}

	public function setPrice(float $price): self
	{
		$this->price = (string) Strings::formatEntityDecimal($price);

		return $this;
	}

	/*******
	 * === Ident
	 */

	public function getIdent(): ?string { return $this->ident; }

	public function setIdent(?string $ident): Payment
	{
		$this->ident = $ident;

		return $this;
	}

	/*******
	 * === Text
	 */

	public function getText(): ?string { return $this->text; }

	public function setText(?string $text): self
	{
		$this->text = $text;

		return $this;
	}

	/*******
	 * === AvailableFrom
	 */

	public function getAvailableFrom(): ?int { return $this->availableFrom; }

	public function setAvailableFrom(int $availableFrom): Payment
	{
		$this->availableFrom = $availableFrom;

		return $this;
	}

	/*******
	 * === AvailableTo
	 */

	public function getAvailableTo(): ?int { return $this->availableTo; }

	public function setAvailableTo(int $availableTo): Payment
	{
		$this->availableTo = $availableTo;

		return $this;
	}

	public function getFreeFrom(): ?int { return $this->freeFrom; }

	public function setFreeFrom(?int $freeFrom): self
	{
		$this->freeFrom = $freeFrom;

		return $this;
	}
}
