<?php declare(strict_types = 1);

namespace EshopOrders\Model\Helpers;

use Brick\Money\Money;
use Core\Model\Helpers\Arrays;
use Core\Model\SystemConfig;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderItemSale;
use EshopOrders\Model\Entities\OrderStatus;
use EshopOrders\Model\EshopOrdersConfig;

class OrderHelper
{
	public static function canSendTrackingUrl(string $statusId, ?Order $order = null): bool
	{
		return Arrays::contains([OrderStatus::STATUS_SPEDITION], $statusId);
	}

	public static function calculateCurrencyPrice(float $price, float $rate, int $decimals): float
	{
		return round($price * (1 / $rate), $decimals);
	}

	public static function removeVat(float $price, ?int $vat): float
	{
		if ((int) $vat === 0) {
			return $price;
		}

		return round($price / (1 + ($vat / 100)), 10);
	}

	public static function checkCountryVatRate(int $vatRate, string $country, bool $validatedVatNumber, ?string $idNumber = null, ?string $vatNumber = null): int
	{
		$country = mb_strtolower($country);
		if ($validatedVatNumber && in_array($country, EshopOrdersConfig::loadArray('invoice.zeroVatIfVatNumberValidatedFor') ?: [], true)) {
			return 0;
		}

		if ($country === 'sk' && $vatRate === 21 && EshopOrdersConfig::load('order.setSkVat20if21')) {
			return 23;
		}

		if ($country === 'cz' && $vatRate === 23) {
			return 21;
		}

		return $vatRate;
	}

	public static function getItemPriceWithoutVat(float $price, int $vatRate): float
	{
		return self::removeVat($price, $vatRate);
	}

	public static function getItemPriceTotalWithoutVat(float $price, int $quantity, int $vatRate, array $sales = []): float
	{
		$price = self::removeVat($price, $vatRate) * $quantity;

		foreach ($sales as $sale) {
			$price -= $sale;
		}

		return $price;
	}

	public static function getItemPriceTotal(float $price, int $quantity, array $sales = []): float
	{
		$price *= $quantity;

		foreach ($sales as $sale) {
			$price -= $sale;
		}

		return $price;
	}

	/**
	 * @param float|int|mixed $value
	 *
	 * @return float|int|mixed
	 */
	public static function calculateSaleValue(string $type, $value, int $quantity, float $basePrice)
	{
		return match ($type) {
			OrderItemSale::TYPE_FIX_ALL => $value,
			OrderItemSale::TYPE_FIX_ONE => $value * $quantity,
			OrderItemSale::TYPE_PERCENT_ALL => (($basePrice * $quantity) / 100) * $value,
			OrderItemSale::TYPE_PERCENT_ONE => (($basePrice / 100) * $value) * $quantity,
			default => $value,
		};
	}

	public static function calculatePriceWithoutVatWithSales(float $totalPrice, float $totalPriceAfterSale, array $dphValues): float
	{
		$result      = 0;
		$hasDiscount = $totalPrice !== $totalPriceAfterSale;

		foreach ($dphValues as $dph => $value) {
			if ($hasDiscount) {
				$result += (($totalPriceAfterSale * ($value / $totalPrice)) / (100 + $dph)) * 100;
			} else {
				$vat = round($value - $value / (1 + ($dph / 100)), 2);

				$result += round($value - $vat, 2);
			}
		}

		return $result;
	}

	public static function calculateVatWithSales(float $totalPrice, float $totalSale, array $dphValues): array
	{
		$result = [];

		foreach ($dphValues as $vat => $value) {
			$price = $value !== 0.0 ? round($totalSale * ($value / $totalPrice), 2) : 0;
			//			bdump($totalSale, (string) $vat);
			//			bdump($value);
			//			bdump($totalPrice);
			//			bdump($price);
			//			bdump($value / $totalPrice);
			$result[$vat]['total'] = -$price;

			$tmpVat = round($price - $price / (1 + ($vat / 100)), 2);

			$result[$vat]['withoutVat'] = -round($price - $tmpVat, 2);
		}

		return $result;
	}

	/**
	 * @param Money             $totalPrice
	 * @param Money             $totalSale
	 * @param array<int, Money> $dphValues
	 *
	 * @return array<int, array<string, float|int>>
	 */
	public static function calculateVatWithSalesMoney(Money $totalPrice, Money $totalSale, array $dphValues): array
	{
		$result = [];

		foreach ($dphValues as $vat => $value) {
			if ($value->getAmount()->toFloat() === 0.0) {
				$price = 0;
			} else {
				$price = round($totalSale->getAmount()->toFloat() * ($value->getAmount()->toFloat() / $totalPrice->getAmount()->toFloat()), 2);
			}

			$result[$vat]['total'] = $price;

			$tmpVat = round($price - $price / (1 + ($vat / 100)), 2);

			$result[$vat]['withoutVat'] = round($price - $tmpVat, 2);
		}

		return $result;
	}

	public static function isOtherCountryVatFilled(?string $country, ?string $vat): bool
	{
		return $country && mb_strtoupper($country) !== SystemConfig::load('defaultCountry') && $vat;
	}

	public static function roundSkCod(float $price): float
	{
		$price = round($price, 2);

		$price *= 100;

		$lastNumber = substr((string) $price, -1);
		$price      = $lastNumber < 5 ? ceil($price / 5) * 5 : round($price / 5) * 5;

		return $price / 100;
	}

	public static function getOrderAttachmentDir(int $orderId): string
	{
		return DATA_DIR . '/eshopOrders/orders/attachments/' . $orderId;
	}
}
