<?php declare(strict_types = 1);

namespace EshopOrders\Model\Listeners;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Event\EventDispatcher;
use Doctrine\Common\EventSubscriber;
use Doctrine\ORM\Events;
use Doctrine\ORM\Event\OnFlushEventArgs;
use Doctrine\ORM\Event\PostFlushEventArgs;
use Doctrine;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderStatus;
use EshopOrders\Model\Event\OrderStatusEvent;
use Override;
use Users\Model\Security\User;
use Users\Model\Users;

class OrderStatusListener implements EventSubscriber
{
	/** @var list<OrderStatus> */
	private array $newOrderStatuses = [];

	/** @var list<OrderStatus> */
	private array $deleteOrderStatuses = [];

	public function __construct(
		protected EntityManagerDecorator $em,
		protected EventDispatcher        $eventDispatcher,
		protected Users                  $usersService,
		protected User                   $user,
	)
	{
	}

	#[Override]
	public function getSubscribedEvents(): array
	{
		return [
			Events::onFlush,
			Events::postFlush,
		];
	}

	public function onFlush(OnFlushEventArgs $event): void
	{
		$uow = $event->getObjectManager()->getUnitOfWork();

		foreach ($uow->getScheduledEntityInsertions() as $entity) {
			if ($entity instanceof OrderStatus) {
				$this->newOrderStatuses[] = $entity;
			}
		}

		foreach ($uow->getScheduledEntityDeletions() as $entity) {
			if ($entity instanceof OrderStatus) {
				$this->deleteOrderStatuses[] = $entity;
			}
		}
	}

	public function postFlush(PostFlushEventArgs $event): void
	{
		if ($this->newOrderStatuses !== []) {
			foreach ($this->newOrderStatuses as $orderStatus) {
				$this->eventDispatcher->dispatch(
					new OrderStatusEvent($orderStatus->getOrder(), $orderStatus->getStatus()->getId()),
					Order::class . '::changeStatus',
				);
			}

			$this->newOrderStatuses = [];
		}

		if ($this->deleteOrderStatuses !== []) {
			foreach ($this->deleteOrderStatuses as $orderStatus) {
				$this->em->getConnection()->executeStatement("INSERT INTO eshop_orders__order_status_history (order_id, status_id, created, created_by) VALUES (?, ?, ?, ?)", [
					$orderStatus->getOrder()->getId(),
					$orderStatus->getStatus()->getId(),
					$orderStatus->getCreated()->format('Y-m-d H:i:s') ?: (new \DateTime())->format('Y-m-d H:i:s'),
					$this->user->getIdentity()?->getId(),
				]);
			}

			$this->deleteOrderStatuses = [];
		}
	}
}
