<?php declare(strict_types = 1);

namespace EshopOrders\Model\Loyalty;

use Core\Model\Entities\EntityManagerDecorator;
use DateTime;
use EshopOrders\Model\CacheService;
use Nette\Utils\Json;
use Tracy\Debugger;

class LoyaltyPointsManager
{
	public const string reasonUse         = 'use';
	public const string reasonOrder       = 'order';
	public const string reasonOrderCancel = 'orderCancel';

	protected array $cLoyaltyId = [];

	public function __construct(
		protected EntityManagerDecorator $em,
		protected CacheService           $cacheService,
	)
	{
	}

	public function getLoyaltyId(int $customerId, string $siteIdent): ?int
	{
		$key = $customerId . '-' . $siteIdent;

		if (!isset($this->cLoyaltyId[$key])) {
			$id = $this->em->getConnection()->fetchOne("SELECT id FROM eshop_orders__customer_loyalty WHERE customer_id = :customerId AND site_id = :siteIdent", [
				'customerId' => $customerId,
				'siteIdent'  => $siteIdent,
			]);

			if ($id) {
				$this->cLoyaltyId[$key] = (int) $id;
			}
		}

		return $this->cLoyaltyId[$key] ?? null;
	}

	public function modify(int $points, int $customerId, string $siteIdent, string $reason, ?int $referenceId = null, ?string $note = null): void
	{
		$conn = $this->em->getConnection();

		$loyaltyId = $this->getLoyaltyId($customerId, $siteIdent);

		try {
			$conn->beginTransaction();
			if (!$loyaltyId) {
				$conn->insert('eshop_orders__customer_loyalty', [
					'customer_id' => $customerId,
					'site_id'     => $siteIdent,
					'points'      => 0,
				]);

				$loyaltyId = $conn->lastInsertId();
			}

			$conn->insert('eshop_orders__customer_loyalty_action', [
				'customer_loyalty_id' => $loyaltyId,
				'points'              => $points,
				'reason'              => $reason,
				'reference_id'        => $referenceId,
				'note'                => $note,
			]);

			$conn->executeQuery("UPDATE eshop_orders__customer_loyalty SET points = points + :points WHERE id = :id", [
				'points' => $points,
				'id'     => $loyaltyId,
			]);
			$conn->executeQuery("UPDATE eshop_orders__customer_loyalty SET last_activity_at = :now WHERE id = :id", [
				'now' => (new DateTime())->format('Y-m-d H:i:s'),
				'id'  => $loyaltyId,
			]);

			$conn->commit();
		} catch (\Exception $e) {
			Debugger::log(Json::encode([
				'points'      => $points,
				'customerId'  => $customerId,
				'siteIdent'   => $siteIdent,
				'reason'      => $reason,
				'referenceId' => $referenceId,
				'note'        => $note,
				'error'       => $e->getMessage(),
			]), 'eshopOrders-loyalty-modify-error');

			if ($conn->isTransactionActive()) {
				$conn->rollBack();
			}
		}

		$this->cacheService->loyaltyCache->clearCustomer($customerId, $siteIdent);
	}

	public function removeAcquiredPointsId(int $orderId, int $customerId, string $siteIdent): void
	{
		$data = $this->getAcquiredPointsByOrderAction($orderId, self::reasonOrder);

		if (!$data) {
			return;
		}

		$conn = $this->em->getConnection();

		try {
			$conn->beginTransaction();

			$conn->delete('eshop_orders__customer_loyalty_action', ['id' => $data['id']]);

			$conn->executeQuery("UPDATE eshop_orders__customer_loyalty SET points = points - :points WHERE id = :id", [
				'points' => $data['points'],
				'id'     => $data['loyaltyId'],
			]);

			$conn->commit();
		} catch (\Exception $e) {
			Debugger::log(Json::encode([
				'id'        => $data['id'],
				'points'    => $data['points'],
				'loyaltyId' => $data['loyaltyId'],
				'error'     => $e->getMessage(),
			]), 'eshopOrders-loyalty-modify-error');

			if ($conn->isTransactionActive()) {
				$conn->rollBack();
			}
		}

		$this->cacheService->loyaltyCache->clearCustomer($customerId, $siteIdent);
	}

	/** @return array{id: int, loyaltyId: int, points: int}|null */
	public function getAcquiredPointsByOrderAction(int $orderId, string $reason): ?array
	{
		$conn = $this->em->getConnection();

		/** @var array{id: int, loyaltyId: int, points: int}|false $data */
		$data = $conn->fetchAssociative("SELECT id, customer_loyalty_id as loyaltyId, points FROM eshop_orders__customer_loyalty_action WHERE reference_id = :orderId AND reason = :reason", [
			'orderId' => $orderId,
			'reason'  => $reason,
		]);

		return $data ?: null;
	}

	public function recalculateCurrentPoints(int $customerId, string $siteIdent): void
	{
		$loyaltyId = $this->getLoyaltyId($customerId, $siteIdent);

		if (!$loyaltyId) {
			return;
		}

		$this->em->getConnection()
			->executeQuery("UPDATE eshop_orders__customer_loyalty SET points = (SELECT SUM(points) FROM eshop_orders__customer_loyalty_action WHERE customer_loyalty_id = :loyaltyId) WHERE id = :loyaltyId", [
				'loyaltyId' => $loyaltyId,
			]);

		$this->cacheService->loyaltyCache->clearCustomer($customerId, $siteIdent);
	}

	public function findPointsToExpireDueInactivity(string $siteIdent, ?int $customerId, int $days): array
	{
		$conn = $this->em->getConnection();

		$expirationDate = (new DateTime())->modify("-{$days} days")->format('Y-m-d H:i:s');

		$where = [
			'cl.site_id = :siteIdent',
			'cl.points > 0',
			'(cl.last_activity_at IS NULL OR cl.last_activity_at < :expirationDate)',
		];

		$params = [
			'siteIdent'      => $siteIdent,
			'expirationDate' => $expirationDate,
		];

		if ($customerId) {
			$where[]              = 'cl.customer_id = :customerId';
			$params['customerId'] = $customerId;
		}

		$result = [];

		foreach ($conn->iterateAssociative("SELECT cl.id, cl.customer_id, cl.points, cl.last_activity_at
				FROM eshop_orders__customer_loyalty cl
				WHERE " . implode(' AND ', $where), $params) as $row) {
			/** @var array $row */
			$customerId   = (int) $row['customer_id'];
			$lastActivity = $row['last_activity_at'] ? DateTime::createFromFormat('Y-m-d H:i:s', $row['last_activity_at']) : null;

			$result[$customerId] = [
				'loyaltyId'      => (int) $row['id'],
				'customerId'     => $customerId,
				'pointsToExpire' => (int) $row['points'],
				'lastActivityAt' => $lastActivity ?: null,
			];
		}

		return $result;
	}
}
