<?php declare(strict_types = 1);

namespace EshopOrders\Model\Loyalty;

use Core\Model\Helpers\Arrays;
use Core\Model\Helpers\Caster;
use Core\Model\Settings;
use EshopOrders\Model\Cache\LoyaltyCache;
use EshopOrders\Model\CacheService;
use EshopOrders\Model\EshopOrdersConfig;
use Nette\Caching\Cache;

class LoyaltySettings
{
	protected static array $isAllowed = [];

	public function __construct(
		protected Settings     $settings,
		protected CacheService $cacheService,
	)
	{
	}

	public function isAllowed(string $siteIdent): bool
	{
		if (!array_key_exists($siteIdent, self::$isAllowed)) {
			self::$isAllowed[$siteIdent] = (bool) $this->getSettings($siteIdent);
		}

		return self::$isAllowed[$siteIdent];
	}

	public function getSettings(string $siteIdent): ?LoyaltySettingsDao
	{
		if (!EshopOrdersConfig::load('allowLoyaltyPoints')) {
			return null;
		}

		$cacheKey = LoyaltyCache::getSettingsKey($siteIdent);

		/** @var LoyaltySettingsDao|null $dao */
		$dao = $this->cacheService->loyaltyCache->cache->load($cacheKey);
		if ($dao === null) {
			$settingsPreview = ucfirst($siteIdent);
			$enabled         = Caster::toBool($this->settings->get($settingsPreview . 'LoyaltyPointsEnable'));

			try {
				if (!$enabled) {
					throw new \Exception('');
				}

				$keys = [
					'registrationPoints',
					'reviewPoints',
					'maxReviewsPerOrder',
					'wholeMoneyValueForPoint',
					'pointsForWholeMoney',
					'minimalOrderValue',
					'maxOrderPercent',
					'oneMoneyForPoints',
					'pointsExpire',
					'emailBeforeExpireDays',
					'name',
					'icon',
				];
				$data = [];

				foreach ($keys as $key) {
					$data[$key] = $this->settings->get($settingsPreview . 'LoyaltyPoints' . ucfirst($key));
				}

				$dao = new LoyaltySettingsDao(
					registrationPoints     : Caster::toInt($data['registrationPoints']),
					reviewPoints           : Caster::toInt($data['reviewPoints']),
					maxReviewsPerOrder     : Caster::toInt($data['maxReviewsPerOrder']),
					wholeMoneyValueForPoint: Caster::toInt($data['wholeMoneyValueForPoint']),
					pointsForWholeMoney    : Caster::toInt($data['pointsForWholeMoney']),
					minimalOrderValue      : Caster::toInt($data['minimalOrderValue']),
					maxOrderPercent        : Caster::toInt($data['maxOrderPercent']),
					oneMoneyForPoints      : Caster::toInt($data['oneMoneyForPoints']),
					pointsExpire           : Caster::toInt($data['pointsExpire']),
					emailBeforeExpireDays  : Caster::toInt($data['emailBeforeExpireDays']),
					name                   : Caster::toString($data['name']),
					icon                   : Caster::toString($data['icon']),
				);

				$this->cacheService->loyaltyCache->cache->save($cacheKey, $dao, [Cache::Expire => '1 month']);

				return $dao;
			} catch (\Exception $e) {
				$this->cacheService->loyaltyCache->cache->save($cacheKey, null, [Cache::Expire => '1 month']);

				return null;
			}
		}

		return $dao;
	}
}
