<?php declare(strict_types = 1);

namespace EshopOrders\Model\Pdf;

use Contributte\Translation\Translator;
use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Event\Event;
use Core\Model\Event\EventDispatcher;
use Core\Model\Helpers\Strings;
use Core\Model\Pdf\Pdf;
use Core\Model\Sites;
use Core\Model\Templating\Template;
use Currency\Model\Currencies;
use EshopCatalog\FrontModule\Model\Sellers;
use EshopOrders\FrontModule\Model\Orders;
use EshopOrders\Model\Entities\Invoice;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Invoices;
use Mpdf\Mpdf;
use Mpdf\MpdfException;
use Mpdf\Output\Destination;

class DeliveryListPdfBuilder
{

	public function __construct(
		protected array                  $orderIds,
		protected Orders                 $ordersService,
		protected Sellers                $sellersService,
		protected Translator             $translator,
		protected Pdf                    $pdfService,
		protected Currencies             $currencies,
		protected EntityManagerDecorator $em,
		protected EventDispatcher        $eventDispatcher,
		protected Invoices               $invoices,
	)
	{
	}

	/**
	 * @return mixed
	 */
	public function render(string $destination = Destination::INLINE)
	{
		$pdf = $this->build();

		return $pdf->Output($this->getFileName(), $destination);
	}

	/**
	 * @throws MpdfException
	 */
	private function build(): Mpdf
	{
		// css
		$customCssFile  = TEMPLATES_DIR . '/Front/default/EshopOrders/_pdf/deliveryListStyle.css';
		$defaultCssFile = SRC_DIR . '/eshoporders/src/Templates/_pdf/deliveryListStyle.css';

		// latte
		$customTemplateFile  = TEMPLATES_DIR . '/Front/default/EshopOrders/_pdf/deliveryList.latte';
		$defaultTemplateFile = SRC_DIR . '/eshoporders/src/Templates/_pdf/deliveryList.latte';

		$pdf = $this->pdfService->newPdf();
		$pdf->SetTopMargin(10);

		// build template
		foreach ($this->orderIds as $orderId) {
			$order = $this->ordersService->getDetail((int) $orderId);
			if (!$order instanceof Order) {
				continue;
			}

			$order->renderPriceDecimals = 2;
			Strings::$priceKeepTail     = true;

			$lang = Sites::$currentLangOverride ?: $order->lang;

			$this->translator->setLocale($lang);

			/** @var Template $template */
			$template = $this->pdfService->getTemplateFactory();
			$template->setFile(file_exists($customTemplateFile) ? $customTemplateFile : $defaultTemplateFile);
			$template->originTemplate = $defaultTemplateFile;

			$template->lang = $lang;

			$seller           = $this->sellersService->getSellerForSite($order->site->getIdent());
			$template->order  = $order;
			$template->seller = $seller;
			$template->addr   = $order->getAddressInvoice();
			$template->bank   = $seller->getBankAccount($order->getCurrencyCode());

			$items                     = $order->getOrderItems()->toArray();
			$event                     = new Event();
			$event->data['orderItems'] = &$items;
			$this->eventDispatcher->dispatch($event, 'eshopOrders.orderItems.loadMoreData');

			if ($order->getInvoice() instanceof Invoice) {
				$invoice = $order->getInvoice();

				$this->invoices->setPriceCalculator($invoice);

				$template->invoice  = $invoice;
				$template->supplier = $invoice->invoiceData->getSupplier();
				$template->bank     = $invoice->invoiceData->getSupplier()->getBank();

				$invoiceData               = $invoice->invoiceData;
				$template->subscriber      = $invoiceData->getCustomer();
				$template->invoiceData     = $invoiceData;
				$template->defaultCurrency = $this->currencies->getDefaultCode();
			}

			$template->originalTemplate = $defaultTemplateFile;
			$template->originalStyles   = $defaultCssFile;

			$pdf->AddPage();
			$pdf->WriteHTML((string) file_get_contents(file_exists($customCssFile) ? $customCssFile : $defaultCssFile), 1);
			$pdf->WriteHTML($template->renderToString(), 0);
		}

		// service config
		$pdf->SetTitle($this->getTitle());

		return $pdf;
	}

	private function getTitle(): string
	{
		return $this->translator->translate('eshopOrdersFront.deliveryList.title');
	}

	public function getFileName(): string
	{
		return sprintf('%s.pdf', Strings::webalize($this->getTitle()));
	}

}
