<?php declare(strict_types = 1);

namespace EshopOrders\Model\Subscribers;

use Contributte\Translation\Translator;
use Core\Components\Flashes\Flashes;
use Core\Model\Entities\EntityManagerDecorator;
use EshopOrders\FrontModule\Model\Affiliate;
use EshopOrders\FrontModule\Model\Event\OrderEvent;
use EshopOrders\Model\Entities\Customer;
use EshopOrders\Model\Entities\Invoice;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderFlag;
use EshopOrders\Model\Entities\OrderStatus;
use EshopOrders\Model\EshopOrdersConfig;
use EshopOrders\Model\Event\EmailEvent;
use EshopOrders\Model\Event\OrderStatusEvent;
use EshopOrders\Model\Invoices;
use EshopOrders\Model\Listeners\InvoiceListener;
use EshopOrders\Model\Loyalty\LoyaltyCalculator;
use EshopOrders\Model\Loyalty\LoyaltyPointsManager;
use EshopOrders\Model\Loyalty\LoyaltySettings;
use EshopOrders\Model\Orders;
use EshopOrders\Model\Pdf\IInvoicePdfBuilderFactory;
use EshopOrders\Model\Statuses;
use Exception;
use Mpdf\Output\Destination;
use Nette\Utils\Html;
use Nette\Utils\Validators;
use Override;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Tracy\Debugger;

class OrderSubscriber implements EventSubscriberInterface
{
	public static bool $enableSendInvoice = false;

	public function __construct(
		protected Translator                $translator,
		protected EntityManagerDecorator    $em,
		protected Statuses                  $statusesService,
		protected Invoices                  $invoices,
		protected IInvoicePdfBuilderFactory $invoicePdfBuilder,
		protected Orders                    $orders,
		protected Affiliate                 $affiliate,
		protected LoyaltyPointsManager      $loyaltyPointsManager,
		protected LoyaltyCalculator         $loyaltyCalculator,
		protected LoyaltySettings           $loyaltySettings,
	)
	{
	}

	#[Override]
	public static function getSubscribedEvents(): array
	{
		return [
			Order::class . '::changeStatus'      => 'orderChangeStatus',
			'eshopOrders.emailChangeOrderStatus' => 'emailChangeOrderStatus',
		];
	}

	public function orderChangeStatus(OrderStatusEvent $event): void
	{
		if ($event->status === OrderStatus::STATUS_IS_PAID) {
			$this->statusesService->changeStatus([$event->order->getId()], OrderStatus::STATUS_IS_PAID);

			$siteIdent = $event->order->site?->getIdent();

			if ($siteIdent && $event->order->getCustomer() && $this->loyaltySettings->isAllowed($siteIdent)) {
				$exist = $this->loyaltyPointsManager->getAcquiredPointsByOrderAction($event->order->getId(), LoyaltyPointsManager::reasonOrder);

				if (!$exist) {
					$points = $this->loyaltyCalculator->calculate($event->order->getPriceItemsDiscount(), $siteIdent);

					if ($points > 0) {
						$this->loyaltyPointsManager->modify($points, $event->order->getCustomer()->getId(), $siteIdent, LoyaltyPointsManager::reasonOrder, $event->order->getId());
					}
				}
			}
		} else if ($event->status === OrderStatus::STATUS_CANCELED) {
			if (EshopOrdersConfig::load('affiliate.enable')) {
				$this->affiliate->cancelOrder($event->order);
			}

			$siteIdent = $event->order->site?->getIdent();

			if ($siteIdent && $event->order->getCustomer() && $this->loyaltySettings->isAllowed($siteIdent)) {
				$exist = $this->loyaltyPointsManager->getAcquiredPointsByOrderAction($event->order->getId(), LoyaltyPointsManager::reasonOrderCancel);

				if (!$exist) {
					if ($event->order->isCorrectiveTaxDocument) {
						// Pokud je storno u ODD, tak body pricitame
						$price = abs($event->order->getPriceItemsDiscount());
					} else {
						// Pokud je storno u normalni objednavky, tak body odecitame
						$price = -$event->order->getPriceItemsDiscount();
					}
					$points = $this->loyaltyCalculator->calculate($price, $siteIdent);

					if ($points > 0) {
						$this->loyaltyPointsManager->modify($points, $event->order->getCustomer()->getId(), $siteIdent, LoyaltyPointsManager::reasonOrderCancel, $event->order->getId());
					}
				}
			}
		}
	}

	public function emailChangeOrderStatus(EmailEvent $event): void
	{
		$order  = $event->order;
		$status = $event->status;

		if (EshopOrdersConfig::load('invoice.enable')
			&& in_array($status, (array) EshopOrdersConfig::load('invoice.orderStatusSendTrigger') ?: [], true)
			&& !$order->hasFlag(OrderFlag::TYPE_INVOICE_SENT)) {
			try {

				if (EshopOrdersConfig::load('invoice.allowUploadCustomInvoiceFile', false)
					&& (!self::$enableSendInvoice || !$order->enableInvoiceGeneration)
					&& !$order->getInvoice()
				) {
					return;
				}

				InvoiceListener::$enableIncreaseStartNumber = false;
				$invoice                                    = $this->invoices->getOneByOrder($this->orders->get((int) $event->order->getId()));
				if (!$invoice instanceof Invoice) {
					$invoice = $this->invoices->getOneByOrder($this->orders->get((int) $event->order->getId()), true);
				}
				$pdfBuilder = $this->invoicePdfBuilder->create($invoice);
				$event->message->addAttachment($pdfBuilder->getFileName(), $pdfBuilder->render(Destination::STRING_RETURN));

				$event->addCustomHtml(Html::el('p')->setText($this->translator->translate('eshopOrdersFront.emails.invoiceIsInAttachment')));

				if (EshopOrdersConfig::load('customerGroup.allowInvoiceCopyToEmail')) {
					$customerGroup = $order->getCustomer() instanceof Customer ? $order->getCustomer()->getGroupCustomers() : null;

					if ($customerGroup && $customerGroup->invoiceCopyEmail) {
						foreach (explode(',', $customerGroup->invoiceCopyEmail) as $email) {
							$email = trim($email);

							if (Validators::isEmail($email)) {
								$event->message->addBcc($email);
							}
						}
					}
				}

				$this->em->getConnection()->insert('eshop_orders__order_flag', [
					'type'     => OrderFlag::TYPE_INVOICE_SENT,
					'state'    => 1,
					'order_id' => $order->getId(),
				]);
			} catch (Exception $e) {
				Flashes::addFlashMessage($this->translator->translate('eshopOrders.order.failedAddInvoiceToEmail'), Flashes::FLASH_DANGER);
			}
		}
	}
}
