<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Components\AdminTools;

use Core\AdminModule\Model\Sites;
use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use EshopOrders\Model\Entities\OrderFlag;
use EshopOrders\Model\EshopOrdersConfig;
use Nette\Utils\Html;

class MarketingInfo extends BaseControl
{
	protected Sites $sites;

	public function __construct(
		Sites $sites
	)
	{
		$this->sites = $sites;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();

		$grid->setDataSource($this->getData());
		$disabledSites = EshopOrdersConfig::load('adminTools.marketingInfo.disableSites') ?: [];

		$sites = [];
		foreach ($this->sites->getOptionsForSelect() as $k => $v) {
			if (in_array($k, $disabledSites, true)) {
				continue;
			}

			$sites[$k] = $v;
		}

		$columnOrder = $grid->addColumnText('order', 'eshopOrders.adminTools.marketingInfoGrid.order')
			->setRenderer(function($row) {
				return Html::el('a', [
					'href'   => $this->presenter->link('Default:editOrder', $row['order']),
					'target' => '_blank',
				])->setText($row['order']);
			});
		$columnOrder->setFilterText();
		$columnSite = $grid->addColumnText('site', 'eshopOrders.adminTools.marketingInfoGrid.site');
		$columnSite->setFilterSelect(['' => ''] + $sites);

		foreach ([
			         OrderFlag::TYPE_QUESTIONING,
			         OrderFlag::TYPE_NEWSLETTER,
			         OrderFlag::typeZboziConversion,
			         OrderFlag::typeEcomailOrder,
		         ] as $flag) {
			$grid->addColumnText($flag, 'eshopOrders.flagsShort.' . $flag)->setRenderer(function($row) use ($flag) {
				$html = Html::el('span', ['class' => 'btn btn-xs']);

				if (isset($row[$flag]) && $row[$flag]['state'] === 1) {
					$html->addClass('btn-success');
					$html->addHtml(Html::el('span', ['class' => 'fa fa-check']));
				} else {
					$html->addClass('btn-danger');
					$html->addHtml(Html::el('span', ['class' => 'fa fa-times']));
				}

				return $html;
			})->setAlign('center');
		}

		$columnOrder->getElementPrototype('th')->addClass('w1nw');

		return $grid;
	}

	protected function getData(): array
	{
		$result        = [];
		$disabledSites = EshopOrdersConfig::load('adminTools.marketingInfo.disableSites') ?: [];

		foreach ($this->em->getConnection()->executeQuery("SELECT o.id, o.site_id, GROUP_CONCAT(CONCAT(f.type, '|', f.state) SEPARATOR ';') as flags FROM eshop_orders__order o
				LEFT JOIN eshop_orders__order_flag f ON o.id = f.order_id " .
			($disabledSites ? "WHERE o.site_id NOT IN ('" . implode("','", $disabledSites) . "') " : '') .
			"GROUP BY o.id
				ORDER BY o.id DESC 
				LIMIT 1000")->iterateAssociative() as $row) {
			/** @var array $row */
			$result[$row['id']] = [
				'order' => $row['id'],
				'site'  => $row['site_id'],
			];

			if ($row['flags']) {
				foreach (explode(';', $row['flags']) as $flagRaw) {
					$flag = explode('|', $flagRaw);

					$result[$row['id']][$flag[0]] = [
						'state' => (int) $flag[1],
					];
				}
			}
		}

		return $result;
	}
}
