<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Components\Order;

use Core\Model\Event\Event;
use Core\Model\Templating\Filters\Price as PriceFilter;
use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use EshopCatalog\Model\Entities\Product;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderItem;
use EshopOrders\Model\EshopOrdersConfig;
use EshopOrders\Model\OrderItems;
use EshopOrders\Model\Orders;
use Nette\Utils\Html;

class OrderItemsGrid extends BaseControl
{
	protected Order       $order;
	protected int         $orderId;
	protected Orders      $ordersService;
	protected OrderItems  $orderItemsService;
	protected PriceFilter $priceFilter;

	public function __construct(
		Order       $order,
		Orders      $orders,
		OrderItems  $orderItems,
		PriceFilter $priceFilter
	)
	{
		$this->order             = $order;
		$this->orderId           = $order->getId();
		$this->ordersService     = $orders;
		$this->orderItemsService = $orderItems;
		$this->priceFilter       = $priceFilter;
	}

	public function render(): void
	{
		if (EshopOrdersConfig::load('orderForm.enableReadyToDeliveryCheckboxes')) {
			$this->template->order = $this->order;
		}

		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();
		$grid->setItemsPerPageList([]);
		$grid->hideTableFooter();

		$prodIds = [];
		foreach ($this->order->getItemsRaw() as $item) {
			if ($item->getProductId()) {
				$prodIds[] = $item->getProductId();
			}
		}

		/** @var Product[] $images */
		$images = $this->em->getRepository(Product::class)->createQueryBuilder('p', 'p.id')
			->select('p, gal, imgs')
			->leftJoin('p.gallery', 'gal')
			->leftJoin('gal.images', 'imgs', 'WITH', 'imgs.isCover = 1 OR imgs.position = 0')
			->andWhere('p.id IN (:ids)')->setParameter('ids', $prodIds)
			->groupBy('p.id')
			->getQuery()
			->getResult();

		$data = [];

		$items = $this->order->getOrderItems();

		/** @var OrderItem[] $items */
		$event                     = new Event();
		$event->data['orderItems'] = &$items;
		$this->eventDispatcher->dispatch($event, 'eshopOrders.orderItems.loadMoreData');

		foreach ($items as $item) {
			$itemId    = $item->getId();
			$productId = $item->getProductId();

			$gifts = [];
			foreach ($item->getGifts() as $gift) {
				$gifts[] = $this->t('eshopOrders.itemsGrid.gift', ['name' => $gift->getName()]);
			}

			$sales = [];
			foreach ($item->sales as $sale) {
				$sales[] = $sale->getFormatted();
			}

			$data[$itemId] = [
				'id'                           => $itemId,
				'productId'                    => $productId,
				'image'                        => '',
				'code1'                        => $item->getCode1() . ($item->getMoreDataValue('stockSupplyAssigned') ? '*' : ''),
				'productName'                  => $item->getOrderItemText()->getName(),
				'productLink'                  => $item->getProduct() && !$item->getProduct()->isDeleted
					? $this->getPresenter()->link(':EshopCatalog:Admin:Products:edit', [$item->getProductId()])
					: null,
				'hasParent'                    => (bool) $item->getParent(),
				'note'                         => $item->getMoreDataValue('note'),
				'gifts'                        => $gifts,
				'quantity'                     => $item->getQuantity(),
				'vatRate'                      => $item->getVatRate(),
				'price'                        => $item->getPrice(),
				'priceCurrency'                => $this->order->currency
					? $item->getPrice(true)
					: null,
				'priceWithoutVat'              => $item->getPriceWithoutVat(),
				'priceWithoutVatCurrency'      => $this->order->currency
					? $item->getPriceWithoutVat(true)
					: null,
				'priceTotal'                   => $item->getPriceTotal(),
				'priceTotalCurrency'           => $this->order->currency
					? $item->getPriceTotal(true)
					: null,
				'priceTotalWithoutVat'         => $item->getPriceTotalWithoutVat(),
				'priceTotalWithoutVatCurrency' => $this->order->currency
					? $item->getPriceTotalWithoutVat(true)
					: null,
				'isReadyToDelivery'            => (bool) $item->isReadyToDelivery,
				'suppliersLinksHtml'           => EshopOrdersConfig::load('orderForm.showSupplierColumn')
					? $item->getSuppliersLinks()
					: null,
				'sales'                        => $sales,
				'attachments'                  => $item->getUploadedFiles(),
			];
		}

		$grid->setDataSource($data);

		//TODO zkontrolovat jestli se neco nacita (treba willcap)
		//		$dataSource->onDataLoaded[] = function($items) {
		//			/** @var OrderItem[] $items */
		//			$event                     = new Event();
		//			$event->data['orderItems'] = &$items;
		//			$this->eventDispatcher->dispatch($event, 'eshopOrders.orderItems.loadMoreData');
		//		};

		//Columns
		if (EshopOrdersConfig::load('orderForm.enableReadyToDeliveryCheckboxes')) {
			$grid->addColumnText('isReadyToDelivery', 'eshopOrders.itemsGrid.readyToDelivery')
				->addCellAttributes(['style' => 'text-align: center !important;'])
				->setTemplateEscaping(false)
				->setRenderer(function(array $row) {
					$el = Html::el('input', [
						'type'    => 'checkbox',
						'name'    => 'isReadyToDelivery_' . $row['id'],
						'checked' => $row['isReadyToDelivery'],
						'style'   => 'transform: scale(1.5); cursor: pointer; width: 15px;',
					]);

					return Html::el('a', [
						'class' => 'ajax',
						'href'  => $this->link(':gridReadyToDeliveryChange!', [
							'id'        => $row['id'],
							'newStatus' => (int) !$row['isReadyToDelivery'],
						]),
					])->setHtml($el);
				});
		}
		$grid->addColumnText('image', 'eshopOrders.defaultGrid.image', '')
			->setRenderer(function(array $row) use ($images) {
				if (!isset($images[$row['productId']]) || !$images[$row['productId']]->getGallery() || !$images[$row['productId']]->getGallery()->getImages()) {
					return '';
				}

				$image = $images[$row['productId']]->getGallery()->getImages()->first();

				if ($image) {
					return Html::el('')
						->addHtml(Html::el('img', [
							'onMouseOver' => "showPicture('spt2-{$row['id']}', 1)",
							'onMouseOut'  => "showPicture('spt2-{$row['id']}', 0)",
							'src'         => $this->imagePipe->request($image->getFilePath(), '50x40'),
						]))
						->addHtml(Html::el('img', [
							'id'    => 'spt2-' . $row['id'],
							'class' => 'show-picture-target',
							'src'   => $this->imagePipe->request($image->getFilePath(), '400x400'),
						]));
				}

				return '';
			})->setAlign('center');
		$grid->addColumnText('code1', 'eshopOrders.itemsGrid.code1');

		if (EshopOrdersConfig::load('orderForm.showSupplierColumn')) {
			$grid->addColumnText('supplier', 'eshopOrders.itemsGrid.supplier', 'suppliersLinksHtml')->setRenderer(function(array $row) {
				return $row['suppliersLinksHtml'];
			});
			$grid->getColumn('supplier')->getElementPrototype('th')->class[] = 'w1nw';
			$grid->getColumn('supplier')->getElementPrototype('td')->class[] = 'w1nw';
		}

		$grid->addColumnText('productName', 'eshopOrders.itemsGrid.productName')
			->setRenderer(function(array $row) {
				$html = Html::el('div');

				$prodId = $row['productId'];
				$link   = $row['productLink']
					? Html::el('a', [
						'href'   => $row['productLink'],
						'target' => '_blank',
					])
					: Html::el('span');

				$productName = $row['productName'];

				if ($row['hasParent']) {
					$html->addClass('ml-3');
					$html->addHtml($link->setHtml('<strong style="color: #f44336; margin-right: 5px">― ― ―</strong> ' . $productName));
				} else {
					$html->addHtml($link->setText($productName));
				}

				if ($row['note']) {
					$html->addHtml(Html::el('div')
						->setHtml($this->t('eshopOrders.itemsGrid.note') . ' - ' . nl2br($row['note'])));
				}

				if ($row['attachments']) {
					$list = Html::el('');

					foreach ($row['attachments'] as $attachment) {
						$list->addHtml(Html::el('div')
							->addHtml(Html::el('a', [
								'href'   => $attachment,
								'target' => '_blank',
							])->setText(basename($attachment))));
					}

					$html->addHtml(Html::el('div')
						->addHtml($list));
				}

				foreach ($row['gifts'] as $gift) {
					$html->addHtml(Html::el('div')
						->setText($gift));
				}

				return $html;
			});

		$grid->addColumnText('quantity', 'eshopOrders.itemsGrid.quantity')
			->setRenderer(fn(array $row) => Html::el('strong')->setText($row['quantity']));

		if (EshopOrdersConfig::load('orderForm.productsVatRateAfterPrice')) {
			$grid->addColumnText('price', 'eshopOrders.itemsGrid.price')
				->setRenderer(function(array $row) {
					return $this->gridRenderPriceSimple(
						$row['price'],
						$row['priceWithoutVat'],
						$this->order->getDefaultCurrency(),
						$row['priceCurrency'],
						$row['priceWithoutVatCurrency'],
						$this->order->currency
							? $this->order->getCurrencyCode()
							: null
					);
				})->setAlign('right');
			$grid->addColumnNumber('vatRate', 'eshopOrders.itemsGrid.vatRate')
				->setFormat(0, ",", " ");
		} else {
			$grid->addColumnNumber('vatRate', 'eshopOrders.itemsGrid.vatRate')
				->setFormat(0, ",", " ");
			$grid->addColumnText('price', 'eshopOrders.itemsGrid.price')
				->setRenderer(function(array $row): Html {
					return $this->gridRenderPriceSimple(
						$row['price'],
						$row['priceWithoutVat'],
						$this->order->getDefaultCurrency(),
						$row['priceCurrency'],
						$row['priceWithoutVatCurrency'],
						$this->order->currency
							? $this->order->getCurrencyCode()
							: null
					);
				})->setAlign('right');
		}

		$grid->addColumnText('sale', 'eshopOrders.itemsGrid.sale')
			->setRenderer(function(array $row) {
				return implode('|', $row['sales']);
			});
		$grid->addColumnText('priceTotal', 'eshopOrders.itemsGrid.priceTotal')
			->setRenderer(function(array $row) {
				return $this->gridRenderPriceSimple(
					$row['priceTotal'],
					$row['priceTotalWithoutVat'],
					$this->order->getDefaultCurrency(),
					$row['priceTotalCurrency'],
					$row['priceTotalWithoutVatCurrency'],
					$this->order->currency
						? $this->order->getCurrencyCode()
						: null
				);
			})->setAlign('right');

		// Actions
		$grid->addAction('front', '', ':EshopCatalog:Front:Default:product', ['id' => 'productId'])
			->setOpenInNewTab()
			->setIcon('link')
			->setBsType('secondary')
			->setRenderCondition(function(array $row) {
				return $row['productId'] && $row['productLink'];
			});
		$grid->addAction('edit', '', ':orderItemForm!', ['itemId' => 'id'])
			->addClass('ajax')
			->setIcon('edit')
			->setBsType('primary');
		$grid->addAction('delete', '', 'orderItemForm:delete!', ['itemId' => 'id'])
			->addClass('ajax')
			->setIcon('times')
			->setBsType('danger')
			->setConfirm('eshopOrders.itemsGrid.deleteConfirm');

		$grid->setColumnsSummary(['priceTotal'])
			->setRenderer(function($row) {
				return $this->gridRenderPriceSimple(
					$this->order->getPriceItems(),
					$this->order->getItemsPriceWithoutVat(),
					$this->order->getDefaultCurrency(),
					$this->order->currency ? $this->order->getPriceItems(true) : null,
					$this->order->currency ? $this->order->getItemsPriceWithoutVat(true) : null,
					$this->order->currency ? $this->order->getCurrencyCode() : null
				)->toHtml();
			});

		// Columns prototype
		$grid->getColumn('image')->getElementPrototype('td')->addClass('w1nw');
		$grid->getColumn('code1')->getElementPrototype('td')->addClass('w1nw');
		$grid->getColumn('quantity')->getElementPrototype('th')->addClass('w1nw');
		$grid->getColumn('vatRate')->getElementPrototype('th')->addClass('w1nw');

		return $grid;
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */
	public function handleGridReadyToDeliveryChange(string $id, string $newStatus): void
	{
		if ($this->orderItemsService->setReadyToDelivery((int) $id, (int) $newStatus)) {
			$this->presenter->flashMessageSuccess('eshopOrders.defaultGrid.publishChanged');
		} else {
			$this->presenter->flashMessageDanger('eshopOrders.defaultGrid.publishChangeFailed');
		}

		$this['grid']->reload();
		$this->presenter->redrawControl('flashes');

		$parent = $this->getParent();
		if ($parent && method_exists($parent, 'redrawControl')) {
			$parent->redrawControl('expeditionForm');
		}
	}

	/*******************************************************************************************************************
	 * =================  Grid function
	 */

	protected function gridRenderPrice(OrderItem $item, string $method): Html
	{
		$order = $item->getOrder();

		if ($order->currency) {
			$price2    = $item->$method(true);
			$code2     = $order->getCurrencyCode();
			$decimals2 = $order->currency->decimals;
		}

		return $this->priceFilter
			->renderTwoPrices($item->$method(), $order->getDefaultCurrency(),
				$price2 ?? null, $code2 ?? null, $decimals2 ?? null,
				$item->{$method . 'WithoutVat'}(), $this->t('eshopOrders.orderForm.noVat'),
				$item->{$method . 'WithoutVat'}(true),
			);
	}

	protected function gridRenderPriceSimple(
		float   $priceVat,
		float   $priceWithoutVat,
		string  $defaultCurrency,
		?float  $priceVatCurrency = null,
		?float  $priceWithoutVatCurrency = null,
		?string $currencyCode = null
	): Html
	{
		return $this->priceFilter->renderTwoPrices(
			$priceVat,
			$defaultCurrency,
			$priceVatCurrency,
			$currencyCode,
			null,
			$priceWithoutVat,
			$this->t('eshopOrders.orderForm.noVat'),
			$priceWithoutVatCurrency
		);
	}
}
