<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Components\Order;

use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderPayment;
use EshopOrders\Model\Orders;
use EshopOrders\Model\Payments;
use Nette\Utils\ArrayHash;

class OrderPaymentForm extends BaseControl
{
	/** @var int @persistent */
	public $orderPaymentId;

	public ?OrderPayment $orderPayment = null;
	public ?Order        $order        = null;

	protected Orders   $ordersService;
	protected Payments $paymentsService;

	public function __construct(
		Orders   $orders,
		Payments $payments
	)
	{
		$this->ordersService   = $orders;
		$this->paymentsService = $payments;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ======================== Components
	 */

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();

		$paymentList = [];
		$othersCountryName = $this->t('admin.others');
		foreach ($this->paymentsService->getAll() as $v) {
			$country = $v->country ? $v->country->getText()->name : $othersCountryName;

			if (!isset($paymentList[$country])) {
				$paymentList[$country] = [];
			}

			$paymentList[$country][$v->getId()] = $v->getName() . ' (cena ' . $v->getPrice() . ' Kč)';
		}

		if (isset($paymentList[$othersCountryName])) {
			$tmp = $paymentList[$othersCountryName];
			unset($paymentList[$othersCountryName]);

			$paymentList[$othersCountryName] = $tmp;
			$tmp                               = null;
		}

		$form->addSelect('payment', 'eshopOrders.default.payment', $paymentList);
		$form->addText('price', 'eshopOrders.default.paymentPrice')
			->setNullable();

		$form->addSaveCancelControl();
		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		try {
			$newPayment = $this->paymentsService->get($values->payment);

			if (!$newPayment) {
				$form->addError('eshopOrders.paymentNotFound');
				$this->redrawControl('form');

				return false;
			}

			$this->paymentsService->setOrderPayment(
				$this->order,
				$newPayment,
				$newPayment->getName(),
				$values->price !== null ? (float) $values->price : null,
			);

			$this->presenter->flashMessageSuccess('eshopOrders.orderPaymentForm.edited');
		} catch (\Exception $e) {
			$form->addError($e->getMessage());
			$this->redrawControl('form');

			return false;
		}

		return true;
	}

	public function setOrder(int $orderId): void
	{
		$this->order        = $this->ordersService->get($orderId);
		$this->orderPayment = $this->order->getPayment();

		if ($this->orderPayment) {
			$i = $this->orderPayment;

			$defaults = ['price' => $i->getPrice()];
			if ($i->getPayment()) {
				$defaults['payment'] = $i->getPayment()->getId();
			}

			$this['form']->setDefaults($defaults);
		}
	}
}
