<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Model;

use Core\Model\Helpers\BaseEntityService;
use EshopOrders\Model\Entities\OrderStatus;
use EshopOrders\Model\Entities\Spedition;
use EshopOrders\Model\EshopOrdersConfig;
use EshopOrders\Model\Helpers\EshopOrdersCache;
use Nette\Caching\Cache;

/**
 * @method Spedition|null getReference($id)
 * @method Spedition|null get($id)
 */
class Speditions extends BaseEntityService
{
	protected $entityClass = Spedition::class;

	protected EshopOrdersCache $eshopOrdersCache;

	protected ?array $cForSelectOption = null;
	protected ?array $cByIdent         = null;

	public function __construct(
		EshopOrdersCache $eshopOrdersCache
	)
	{
		$this->eshopOrdersCache = $eshopOrdersCache;
	}

	/**
	 * @return Spedition[]
	 */
	public function getAll(): array
	{
		return $this->getEr()
			->createQueryBuilder('s', 's.id')
			->orderBy('s.position', 'asc')
			->getQuery()
			->getResult();
	}

	public function getByIdent(string $ident): ?Spedition
	{
		if ($this->cByIdent === null) {
			$this->cByIdent = [];

			foreach ($this->getEr()->findAll() as $row) {
				/** @var Spedition $row */
				$this->cByIdent[$row->getIdent()] = $row;
			}
		}

		return $this->cByIdent[$ident] ?? null;
	}

	public function setPosition(int $id, int $position): bool
	{
		if ($item = $this->get($id)) {
			$item->setPosition($position);
			$this->em->persist($item);
			$this->em->flush();

			return true;
		}

		return false;
	}

	public function setPublish(int $id, int $state): bool
	{
		if ($item = $this->getReference($id)) {
			$item->isPublished = $state;
			$this->em->persist($item);
			$this->em->flush();

			return true;
		}

		return false;
	}

	public function getForSelectOption(): array
	{
		if ($this->cForSelectOption === null) {
			$this->cForSelectOption = $this->getEr()->findPairs([], 'name') ?: [];
		}

		return $this->cForSelectOption;
	}

	/**
	 * Vrátí ID objednávek které lze exportovat. Je nutná následná filtrace.
	 */
	public function checkOrdersGridMenuExportOnlyInStatus(array $ids): array
	{
		$onlyInStatus = EshopOrdersConfig::load('ordersGridMenuExportOnlyInStatus', null);

		if ($onlyInStatus === null) {
			return $ids;
		}

		$allowed = [];
		foreach ($this->em->getRepository(OrderStatus::class)->createQueryBuilder('os')
			         ->select('IDENTITY(os.order) as order')
			         ->where('os.order IN (:order)')->setParameter('order', $ids)
			         ->andWhere('os.status = :status')->setParameter('status', $onlyInStatus)
			         ->andWhere('os.deleted IS NULL')
			         ->groupBy('os.order')->getQuery()->getArrayResult() as $row) {
			$allowed[] = $row['order'];
		}

		return $allowed;
	}

	public function remove($id)
	{
		$result = parent::remove($id);

		if ($result) {
			$this->eshopOrdersCache->getCache()->remove('speditionWeights');
			$this->eshopOrdersCache->getCache()->clean([
				Cache::Tags => ['spedition', 'freeFrom'],
			]);
		}

		return $result;
	}

	public function getAllAssignedCountries(): array
	{
		$result = [];
		$conn   = $this->em->getConnection();

		foreach ($conn->fetchFirstColumn("SELECT country_id FROM eshop_orders__payment_spedition_country GROUP BY country_id") as $row) {
			$result[] = $row;
		}

		return $result;
	}
}
