<?php declare(strict_types = 1);

namespace EshopOrders\Console\Payment;

use Contributte\Translation\Translator;
use Core\Model\Event\EventDispatcher;
use Doctrine;
use Doctrine\ORM\Tools\Console\Command\SchemaTool\AbstractCommand;
use EshopOrders\FrontModule\Model\Event\OrderEvent;
use EshopOrders\Model\Entities\OrderCardPaymentHistory;
use EshopOrders\Model\Entities\OrderHistory;
use EshopOrders\Model\Orders;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;

class RemindCardPayment extends AbstractCommand
{
	/** @var Orders|null @inject */
	public ?Orders $orders = null;

	/** @var EventDispatcher|null @inject */
	public ?EventDispatcher $eventDispatcher = null;

	/** @var Translator|null @inject */
	public ?Translator $translator = null;

	protected function configure(): void
	{
		$this->setName('eshoporders:payment:remind-card')->setDescription('Remind failed card payment');
	}

	protected function executeSchemaCommand(InputInterface $input, OutputInterface $output, Doctrine\ORM\Tools\SchemaTool $schemaTool, array $metadatas, SymfonyStyle $ui)
	{
		$output->writeln('Remind failed card payment');

		try {
			$em   = $this->orders->em;
			$conn = $em->getConnection();

			$yesterday = (new \DateTime())->modify('-1 day')->format('Y-m-d');
			$orders    = [];

			foreach ($conn->fetchAllAssociative("SELECT o.id FROM eshop_orders__order o
       				INNER JOIN eshop_orders__order_card_payment_token t ON o.id = t.order_id AND t.id IN (
	                    SELECT h.token_id FROM eshop_orders__order_card_payment_history h
						WHERE h.created >= :from AND h.created <= :to AND h.status = :statusCreated AND h.token_id IN (
						    SELECT h2.token_id
						    FROM eshop_orders__order_card_payment_history h2
						    GROUP BY h2.token_id
						    HAVING SUM(CASE WHEN h2.status = :statusPaid THEN 1 ELSE 0 END) = 0
	                    )
					)
					WHERE o.is_paid = 0
       				", [
				'statusCreated' => OrderCardPaymentHistory::STATUS_CREATED,
				'statusPaid'    => OrderCardPaymentHistory::STATUS_PAID,
				'from'          => $yesterday . ' 00:00:00',
				'to'            => $yesterday . ' 23:59:59',
			]) as $row) {
				$orders[] = (int) $row['id'];
			}

			$ordersCount = count($orders);
			$i           = 1;
			foreach ($orders as $orderId) {
				$order = $this->orders->getFullOrder($orderId);

				if ($order) {
					$this->eventDispatcher->dispatch(new OrderEvent($order), 'eshopOrders.sendCardPaymentReminder');

					$orderHistory = new OrderHistory($order, $this->translator->translate('eshopOrders.paymentReminderSent'), null);

					$em->persist($orderHistory);
					$em->flush();

					$em->clear();
					gc_collect_cycles();

					if ($i < $ordersCount) {
						sleep(15);
					}
				}
			}

			$output->writeLn('<info>[OK]</info>');
		} catch (\Exception $e) {
			$output->writeln('<error>[ERROR] - ' . $e->getMessage() . '</error>');
		}

		return 0;
	}
}
