<?php declare(strict_types = 1);

namespace EshopOrders\CronModule\Model;

use Core\Model\Entities\EntityManagerDecorator;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderStatus;
use EshopOrders\Model\Orders as OrdersService;

class Orders
{
    protected EntityManagerDecorator $em;
    protected OrdersService $ordersService;

    public function __construct(
        EntityManagerDecorator $em,
        OrdersService          $ordersService
    )
    {
        $this->em            = $em;
        $this->ordersService = $ordersService;
    }

	public function ordersByDate(string $siteIdent, string $from, string $to, ?string $lang = null): array
	{
		$orders = [];

		$params = [
			'siteIdent' => $siteIdent,
			'status'    => 'created',
			'from'      => $from . ' 00:00:00',
			'to'        => $to . ' 23:59:59',
		];

		$orderJoin = [
			'o.id = os.order_id',
			'o.site_id = :siteIdent',
		];

		$andWhere = [
			'os.status_id = :status',
			'os.created >= :from',
			'os.created <= :to',
		];

		if ($lang) {
			$orderJoin[]    = 'o.lang = :lang';
			$params['lang'] = $lang;
		}

		$cancelled = [];
		$ids       = $this->em->getConnection()->executeQuery("SELECT os.order_id FROM eshop_orders__order_status os 
                INNER JOIN eshop_orders__order o ON " . implode(' AND ', $orderJoin) . "
                WHERE " . implode(' AND ', $andWhere), $params)->fetchFirstColumn();

		if ($ids === []) {
			return [];
		}

		foreach (array_chunk($ids, 200) as $chunk) {
			foreach ($this->em->getConnection()->executeQuery("SELECT os.order_id FROM eshop_orders__order_status os
				WHERE os.status_id = :status", [
				'status' => OrderStatus::STATUS_CANCELED,
			])->fetchFirstColumn() as $id) {
				$cancelled[] = $id;
			}
		}

		$i = 0;
		foreach (array_diff($ids, $cancelled) as $id) {
			/** @var ?Order $order */
			$order = $this->ordersService->getFullOrder($id);
			if (!$order) {
				continue;
			}

			$data = [
				'id'                               => $order->getId(),
				'created'                          => $order->getCreatedTime() ? $order->getCreatedTime()->format('Y-m-d H:i:s') : null,
				'priceWithVat'                     => $order->getPrice(),
				'priceWithoutVat'                  => $order->getPriceWithoutVat(),
				'itemsPriceWithVatWithDiscount'    => $order->getPriceItemsDiscount(),
				'itemsPriceWithoutVatWithDiscount' => $order->getPriceItemsWithoutVatDiscount(),
				'company'                          => '',
				'firstName'                        => '',
				'lastName'                         => '',
				'email'                            => '',
				'phone'                            => '',
				'customerId'                       => '',
				'customerGroup'                    => '',
				'customerType'                     => 'B2C',
				'payment'                          => '',
				'paymentId'                        => '',
				'paymentCost'                      => 0,
				'spedition'                        => '',
				'speditionId'                      => '',
				'speditionCost'                    => 0,
				'current'                          => $order->getCurrencyCode(),
			];

			$addrInv = $order->getAddressInvoice();
			if ($addrInv) {
				$data['company']   = $addrInv->getCompany();
				$data['firstName'] = $addrInv->getFirstName();
				$data['lastName']  = $addrInv->getLastName();
				$data['email']     = $addrInv->getEmail();
				$data['phone']     = $addrInv->getPhone();
			}

			$customer = $order->getCustomer();
			if ($customer) {
				$data['customerId'] = $customer->getId();

				if ($customer->getGroupCustomers()) {
					$data['customerGroup'] = $customer->getGroupCustomers()->name;
					$data['customerType']  = $customer->getGroupCustomers()->type ?: 'B2C';
				}
			}

			if ($order->getPayment()) {
				$data['payment']     = $order->getPayment()->getPayment() ? $order->getPayment()->getPayment()->getName() : $order->getPayment()->getName();
				$data['paymentId']   = $order->getPayment()->getPaymentId();
				$data['paymentCost'] = $order->getPayment()->getPrice();
			}

			if ($order->getSpedition()) {
				$data['spedition']     = $order->getSpedition()->getSpedition() ? $order->getSpedition()->getSpedition()->getName() : $order->getSpedition()->getName();
				$data['speditionId']   = $order->getSpedition()->getSpeditionId();
				$data['speditionCost'] = $order->getSpedition()->getPrice();
			}

			$orders[] = $data;

			$i++;

			if ($i % 10 === 0) {
				$this->em->clear();
				gc_collect_cycles();
			}
		}

		$this->em->clear();
		gc_collect_cycles();

		return $orders;
	}
}
