<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Components\Customer;

use Core\Model\Countries;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use EshopOrders\FrontModule\Model\Customers;
use EshopOrders\Model\Entities\Customer;
use EshopOrders\Model\Entities\CustomerAddress;
use EshopOrders\Model\EshopOrdersConfig;
use Nette\Forms\Form;
use Nette\Utils\ArrayHash;
use Users\Model\Entities\User;

class ContactData extends BaseControl
{
	public User        $user;
	protected Customer $customer;

	protected Customers $customers;
	protected Countries $countriesService;

	public function __construct(
		User      $user,
		Customers $customers,
		Countries $countries
	)
	{
		$this->customers        = $customers;
		$this->countriesService = $countries;
		$this->user             = $user;
		$this->customer         = $this->customers->getOrCreateCustomer($user->getId(), $user->getEmail(), $user->getName(), $user->getLastname());
	}

	public function render(): void
	{

		$this->template->addressesOptions = EshopOrdersConfig::load('allowAddressManagement') ? $this->getAddressesOptions() : [];
		$this->template->customer         = $this->customer;
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax();
		$d = [];

		if (!EshopOrdersConfig::load('allowAddressManagement')) {
			$invContainer = $form->addContainer('invoice');
			$invContainer->addText('firstName', 'eshopOrdersFront.contactData.firstName');
			$invContainer->addText('lastName', 'eshopOrdersFront.contactData.lastName');
			$invContainer->addPhone('phone', 'eshopOrdersFront.contactData.phone');
			$invContainer->addEmail('email', 'eshopOrdersFront.contactData.email');
			$invContainer->addText('street', 'eshopOrdersFront.contactData.street');
			$invContainer->addText('city', 'eshopOrdersFront.contactData.city');
			$invContainer->addText('postal', 'eshopOrdersFront.contactData.postal');
			$invContainer->addSelect('country', 'eshopOrdersFront.contactData.country', $this->countriesService->getAllNameColumn());
			$invContainer->addCheckbox('isCompany', 'eshopOrdersFront.contactData.isCompany');
			$invCompany   = $invContainer->addText('company', 'eshopOrdersFront.contactData.company');
			$invIdNumber  = $invContainer->addText('idNumber', 'eshopOrdersFront.contactData.idNumber');
			$invVatNumber = $invContainer->addText('vatNumber', 'eshopOrdersFront.contactData.vatNumber');
			if (EshopOrdersConfig::load('customer.showIdVatNumber', false)) {
				$invIdVatNumber = $invContainer->addText('idVatNumber', 'eshopOrdersFront.contactData.idVatNumber');
			}

			$useDeliAddr   = $form->addCheckbox('useDeliveryAddress', 'eshopOrdersFront.contactData.useDeliveryAddress');
			$deliContainer = $form->addContainer('delivery');
			$deliContainer->addText('firstName', 'eshopOrdersFront.contactData.firstName');
			$deliContainer->addText('lastName', 'eshopOrdersFront.contactData.lastName');
			$deliContainer->addPhone('phone', 'eshopOrdersFront.contactData.phone');
			$deliContainer->addText('email', 'eshopOrdersFront.contactData.email')
				->addConditionOn($useDeliAddr, Form::FILLED)
				->addRule(Form::EMAIL);
			$deliContainer->addText('street', 'eshopOrdersFront.contactData.street');
			$deliContainer->addText('city', 'eshopOrdersFront.contactData.city');
			$deliContainer->addText('postal', 'eshopOrdersFront.contactData.postal');
			$deliContainer->addSelect('country', 'eshopOrdersFront.contactData.country', $this->countriesService->getAllNameColumn());
			$deliContainer->addCheckbox('isCompany', 'eshopOrdersFront.contactData.isCompany');
			$deliContainer->addText('company', 'eshopOrdersFront.contactData.company');
			$deliContainer->addText('idNumber', 'eshopOrdersFront.contactData.idNumber');
			$deliContainer->addText('vatNumber', 'eshopOrdersFront.contactData.vatNumber');

			if (EshopOrdersConfig::load('customer.showIdVatNumber', false)) {
				$deliContainer->addText('idVatNumber', 'eshopOrdersFront.contactData.idVatNumber');
			}

			$customerAddrInv = $this->customer->getAddressInvoice();
			if ($customerAddrInv) {
				$d += [
					'invoice' => [
						'firstName'   => $customerAddrInv->getFirstName(),
						'lastName'    => $customerAddrInv->getLastName(),
						'email'       => $customerAddrInv->email,
						'phone'       => $customerAddrInv->phone,
						'street'      => $customerAddrInv->street,
						'city'        => $customerAddrInv->city,
						'postal'      => $customerAddrInv->postal,
						'country'     => $customerAddrInv->getCountry() ? $customerAddrInv->getCountry()->getId() : null,
						'isCompany'   => $customerAddrInv->company || $customerAddrInv->idNumber || $customerAddrInv->vatNumber,
						'company'     => $customerAddrInv->company,
						'idNumber'    => $customerAddrInv->idNumber,
						'vatNumber'   => $customerAddrInv->vatNumber,
						'idVatNumber' => $customerAddrInv->idVatNumber,
					],
				];
			}

			$customerAddrDelivery = $this->customer->getAddressDelivery();
			if ($customerAddrDelivery) {
				$d += [
					'useDeliveryAddress' => $customerAddrDelivery->email || $customerAddrDelivery->firstName || $customerAddrDelivery->company,
					'delivery'           => [
						'firstName'   => $customerAddrDelivery->getFirstName(),
						'lastName'    => $customerAddrDelivery->getLastName(),
						'email'       => $customerAddrDelivery->email,
						'phone'       => $customerAddrDelivery->phone,
						'street'      => $customerAddrDelivery->street,
						'city'        => $customerAddrDelivery->city,
						'postal'      => $customerAddrDelivery->postal,
						'country'     => $customerAddrDelivery->getCountry() ? $customerAddrDelivery->getCountry()->getId() : null,
						'isCompany'   => $customerAddrDelivery->company || $customerAddrDelivery->idNumber || $customerAddrDelivery->vatNumber,
						'company'     => $customerAddrDelivery->company,
						'idNumber'    => $customerAddrDelivery->idNumber,
						'vatNumber'   => $customerAddrDelivery->vatNumber,
						'idVatNumber' => $customerAddrDelivery->idVatNumber,
					],
				];
			}

			if (EshopOrdersConfig::load('customer.disableModifyInvoiceAddress')) {
				foreach ($invContainer->getComponents() as $comp) {
					/** @var \Nette\Forms\Controls\BaseControl $comp */
					$comp->setOmitted(true);
					$comp->setHtmlAttribute('data-omitted', 'true');
				}
			} else if (EshopOrdersConfig::load('customer.disableModifyIco')) {
				$invCompany->setOmitted(true)
					->setHtmlAttribute('data-omitted', 'true');
				$invIdNumber->setOmitted(true)
					->setHtmlAttribute('data-omitted', 'true');
				$invVatNumber->setOmitted(true)
					->setHtmlAttribute('data-omitted', 'true');
				if (isset($invIdVatNumber)) {
					$invIdVatNumber->setOmitted(true)
						->setHtmlAttribute('data-omitted', 'true');
				}
			}
		} else {
			$customer  = $this->customer;
			$addresses = [];

			foreach ($customer->addressesAvailable as $addr) {
				if (EshopOrdersConfig::load('customer.disableModifyInvoiceAddress')
					&& $customer->getAddressInvoice()
					&& $customer->getAddressInvoice()->getId() === $addr->getId()
				) {
					continue;
				}

				$addresses[$addr->getId()] = $addr->getDetailString();
			}

			$form->addSelect('addressDelivery', 'eshopOrdersFront.customer.defaultAddressDelivery', $addresses);

			if (!EshopOrdersConfig::load('customer.disableModifyInvoiceAddress')) {
				$form->addSelect('addressInvoice', 'eshopOrdersFront.customer.defaultAddressInvoice', $addresses);
			}
		}

		$form->addSubmit('submit', 'default.save')
			->setBtnClass('btn-success primary-bg-color');

		$form->setDefaults($d);

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(Form $form, ArrayHash $values): bool
	{
		try {
			$user     = $this->user;
			$customer = $this->customer;

			if (!EshopOrdersConfig::load('allowAddressManagement', false)) {
				$customerAddrInv = $this->customer->getAddressInvoice();
				if (!$customerAddrInv) {
					$customerAddrInv = new CustomerAddress($customer);
				}

				if (!EshopOrdersConfig::load('customer.disableModifyInvoiceAddress')) {
					$formInvData                = $values->invoice;
					$customerAddrInv->firstName = $formInvData->firstName;
					$customerAddrInv->lastName  = $formInvData->lastName;
					$customerAddrInv->email     = $formInvData->email;
					$customerAddrInv->phone     = $formInvData->phone;
					$customerAddrInv->street    = $formInvData->street;
					$customerAddrInv->city      = $formInvData->city;
					$customerAddrInv->postal    = $formInvData->postal;

					if (!EshopOrdersConfig::load('customer.disableModifyIco')) {
						$customerAddrInv->company   = $formInvData->isCompany ? $formInvData->company : null;
						$customerAddrInv->idNumber  = $formInvData->isCompany ? $formInvData->idNumber : null;
						$customerAddrInv->vatNumber = $formInvData->isCompany ? $formInvData->vatNumber : null;

						if (EshopOrdersConfig::load('customer.showIdVatNumber', false)) {
							$customerAddrInv->idVatNumber = $formInvData->isCompany ? $formInvData->idVatNumber : null;
						}
					}
				}

				$this->em->persist($customerAddrInv);

				$customerAddrDeli = $this->customer->getAddressDelivery();
				if (!$customerAddrDeli) {
					$customerAddrDeli = new CustomerAddress($customer);
				}

				if (!$values->useDeliveryAddress) {
					$customerAddrDeli->clearFields();
				} else {
					$formDeliData                = $values->delivery;
					$customerAddrDeli->firstName = $formDeliData->firstName;
					$customerAddrDeli->lastName  = $formDeliData->lastName;
					$customerAddrDeli->email     = $formDeliData->email;
					$customerAddrDeli->phone     = $formDeliData->phone;
					$customerAddrDeli->street    = $formDeliData->street;
					$customerAddrDeli->city      = $formDeliData->city;
					$customerAddrDeli->postal    = $formDeliData->postal;
					$customerAddrDeli->company   = $formDeliData->company;
					$customerAddrDeli->idNumber  = $formDeliData->idNumber;
					$customerAddrDeli->vatNumber = $formDeliData->vatNumber;
					if (EshopOrdersConfig::load('customer.showIdVatNumber', false)) {
						$customerAddrDeli->idVatNumber = $formDeliData->idVatNumber;
					}
				}

				$this->em->persist($customerAddrDeli);
			} else {
				/** @var CustomerAddress|null $addrRef */
				$addrRef = $values->addressDelivery ? $this->em->getReference(CustomerAddress::class, $values->addressDelivery) : null;

				$customer->setAddressDelivery($addrRef);

				if (!EshopOrdersConfig::load('customer.disableModifyInvoiceAddress')) {
					/** @var CustomerAddress|null $addrRef */
					$addrRef = $values->addressInvoice ? $this->em->getReference(CustomerAddress::class, $values->addressInvoice) : null;

					$customer->setAddressInvoice($addrRef);
				}
			}

			$this->em->flush();
			$this->presenter->flashMessageSuccess('eshopOrdersFront.contactData.updated');
		} catch (\Exception $e) {
			$form->addError($e->getMessage());
			$this->redrawControl('form');

			return false;
		}

		$this->redirect('this');
	}

	public function getAddressesOptions(): array
	{
		$addresses = [
			'delivery' => '',
			'invoice'  => '',
		];

		$customer = $this->customer;

		$deliveryDefault = $customer->getAddressDelivery() ? $this->customer->getAddressDelivery()->getId() : null;
		$invoiceDefault  = $customer->getAddressInvoice() ? $this->customer->getAddressInvoice()->getId() : null;

		foreach ($this->customer->addressesAvailable as $addr) {
			if (EshopOrdersConfig::load('customer.disableModifyInvoiceAddress')
				&& $invoiceDefault === $addr->getId()) {
				continue;
			}

			$str                   = $addr->getDetailString();
			$addresses['delivery'] .= '<option value="' . $addr->getId() . '"' . ($deliveryDefault === $addr->getId() ? ' selected' : '') . '>' . $addr->getDetailString() . '</option>';
			$addresses['invoice']  .= '<option value="' . $addr->getId() . '"' . ($invoiceDefault === $addr->getId() ? ' selected' : '') . '>' . $addr->getDetailString() . '</option>';
		}

		return $addresses;
	}
}

