<?php declare(strict_types=1);

namespace EshopOrders\FrontModule\Components\Customer;

use Core\Model\UI\BaseControl;
use EshopOrders\FrontModule\Model\Customers;
use EshopOrders\FrontModule\Model\OrderFacade;
use EshopOrders\Model\Entities\Customer;
use EshopOrders\Model\Orders;
use Nette\Utils\Paginator;

class PurchasedProducts extends BaseControl
{
    /** @persistent */
    public int $page = 1;

    public int $limit = 5;

    protected Orders      $ordersService;
    protected Customers   $customersService;
    protected OrderFacade $orderFacade;

    protected ?Customer  $customer  = null;
    protected ?Paginator $paginator = null;

    public function __construct(
        Orders      $orders,
        Customers   $customers,
        OrderFacade $orderFacade
    )
    {
        $this->ordersService    = $orders;
        $this->customersService = $customers;
        $this->orderFacade      = $orderFacade;
    }

    public function render(): void
    {
        $customer = $this->getCustomer();

        if (!$customer) {
            return;
        }

        $paginator = $this->getPaginator();
        $orders    = $this->ordersService->getByCustomer($customer, $paginator->getItemsPerPage(), $paginator->getOffset());

        $this->template->orders    = $orders;
        $this->template->paginator = $paginator;
        $this->template->render($this->getTemplateFile());
    }

    public function handlePaginator(int $page): void
    {
        $this->page = $page;

        $this->redrawControl('list');
        $this->redrawControl('paginator');
    }

    public function getPaginator(): Paginator
    {
        if ($this->paginator === null) {
            $itemsPerPage = $this->limit;

            $paginator = new Paginator();
            $paginator->setItemCount($this->getCustomer() ? $this->ordersService->getOrdersCountByCustomer($this->getCustomer()) : 0);
            $paginator->setItemsPerPage($itemsPerPage);
            $paginator->setPage($this->page);

            $this->paginator = $paginator;
        }

        return $this->paginator;
    }

    protected function getCustomer(): ?Customer
    {
        if ($this->customer === null) {
            $userId         = (int) $this->getPresenter()->getUser()->id;
            $this->customer = $this->customersService->getByUserId($userId);
        }

        return $this->customer;
    }

}