<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Model;

use Core\Model\Entities\EntityManagerDecorator;
use DateTime;
use EshopOrders\Model\Entities\Affiliate\AffiliateCampaign;
use EshopOrders\Model\Entities\Affiliate\AffiliateOrder;
use EshopOrders\Model\Entities\Affiliate\AffiliatePartner;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\EshopOrdersConfig;
use Nette\Caching\Cache;
use Nette\Http\Request;
use Nette\Http\Response;
use Nette\Utils\Json;
use Tracy\Debugger;

class Affiliate
{
	protected EshopOrdersCache       $eshopOrdersCache;
	protected EntityManagerDecorator $em;
	protected Request                $request;
	protected Response               $response;

	public function __construct(
		EshopOrdersCache       $eshopOrdersCache,
		EntityManagerDecorator $em,
		Request                $request,
		Response               $response
	)
	{
		$this->eshopOrdersCache = $eshopOrdersCache;
		$this->em               = $em;
		$this->request          = $request;
		$this->response         = $response;
	}

	public function getBaseData(): array
	{
		return $this->eshopOrdersCache->getAffiliateCache()->load('affiliateBase', function(&$dep) {
			$dep = [
				Cache::EXPIRE => '1 week',
			];

			$partners = $this->em->getRepository(AffiliatePartner::class)
				->createQueryBuilder('ap', 'ap.id')
				->select('ap.id, ap.ident')
				->getQuery()->getArrayResult();

			$result = [];
			foreach ($this->em->getRepository(AffiliateCampaign::class)
				         ->createQueryBuilder('ac', 'ac.ident')
				         ->select('ac.id, ac.ident, ac.commission, IDENTITY(ac.partner) as partnerId')
				         ->getQuery()->getScalarResult() as $row) {
				$partner      = $partners[$row['partnerId']];
				$partnerIdent = $partner['ident'];

				if (!isset($result[$partnerIdent])) {
					$result[$partnerIdent] = [];
				}

				$result[$partnerIdent][$row['ident']] = [
					'partnerId'     => $partner['id'],
					'partnerIdent'  => $partnerIdent,
					'campaignId'    => $row['id'],
					'campaignIdent' => $row['ident'],
					'commission'    => $row['commission'],
				];
			}

			return $result;
		});
	}

	public function checkCookie(): ?array
	{
		try {
			$affiliateCookie = $this->request->getCookie('affiliate');

			if ($affiliateCookie) {
				$affiliateData = Json::decode(base64_decode($affiliateCookie), Json::FORCE_ARRAY);

				if (isset($affiliateData['partnerId'], $affiliateData['campaignId'])) {
					return $affiliateData;
				}
			}
		} catch (\Exception $e) {
		}

		return null;
	}

	public function addOrder(int $campaignId, Order $order): void
	{
		if (!EshopOrdersConfig::load('affiliate.enable')) {
			return;
		}

		/** @var ?AffiliateCampaign $campaign */
		$campaign = $this->em->getRepository(AffiliateCampaign::class)->find($campaignId);
		if (!$campaign) {
			return;
		}

		$value = round($order->getPriceItemsDiscount() * ($campaign->commission / 100), 2);

		$affiliateOrder = new AffiliateOrder($order, $campaign, $value, $order->getPriceItemsDiscount());

		if ($order->getCustomer()) {
			$affiliateOrder->email = $order->getCustomer()->getUser()->getEmail();
		} else if ($order->getAddressInvoice()) {
			$affiliateOrder->email = $order->getAddressInvoice()->getEmail();
		}

		$this->em->persist($affiliateOrder);
		$this->em->flush();

		$this->response->deleteCookie('affiliate');
	}

	public function cancelOrder(Order $order): void
	{
		if (!EshopOrdersConfig::load('affiliate.enable')) {
			return;
		}

		$exist = $this->em->getConnection()->fetchOne("SELECT id FROM eshop_orders__affiliate_order WHERE order_id = :orderId", ['orderId' => $order->getId()]);

		if ($exist) {
			$this->em->getConnection()->update('eshop_orders__affiliate_order', [
				'cancelled' => (new \DateTime())->format('Y-m-d H:i:s'),
			], [
				'id' => $exist,
			]);
		}
	}

	public function addClick(int $campaignId): void
	{
		try {
			$statId = $this->getCurrentStatIdByCampaign($campaignId);

			$this->em->getConnection()->executeQuery("UPDATE eshop_orders__affiliate_campaign_stat SET clicks = clicks + 1 WHERE id = ?", [$statId]);
		} catch (\Exception $e) {
			Debugger::log($e, 'affiliate-stat');
		}
	}

	public function getCurrentStatIdByCampaign(int $campaignId): int
	{
		$today = (new DateTime)->format('Y-m-d');
		$key   = 'campaignStatId/' . $campaignId . '/' . $today;

		$cache  = $this->eshopOrdersCache->getAffiliateCache();
		$statId = $cache->load($key, function(&$dep) use ($campaignId, $today) {
			$dep[Cache::EXPIRE] = '1 hour';

			return $this->em->getConnection()->fetchOne("SELECT id FROM eshop_orders__affiliate_campaign_stat WHERE campaign_id = :campaign AND date = :date", [
				'campaign' => $campaignId,
				'date'     => $today,
			]) ?? false;
		});

		if (!$statId) {
			$this->em->getConnection()->insert('eshop_orders__affiliate_campaign_stat', [
				'date'        => $today,
				'clicks'      => 0,
				'campaign_id' => $campaignId,
			]);

			$statId = $this->em->getConnection()->lastInsertId();
			$cache->save($key, $statId);
		}

		return (int) $statId;
	}
}
