<?php declare(strict_types = 1);

namespace EshopOrders\FrontModule\Presenters;

use Core\Model\Event\PresenterTemplateEvent;
use EshopCatalog\FrontModule\Model\Sellers;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Helpers\QrGenerator;
use EshopOrders\Model\Orders;
use EshopOrders\Model\Payments;
use EshopOrders\Model\Statuses;
use Navigations\FrontModule\Model\LinksHelper;
use Nette\Application\BadRequestException;
use Nette\Application\Responses\FileResponse;
use Nette\Http\Url;
use Nette\Utils\FileSystem;
use Nette\Utils\Image;

class PaymentPresenter extends BasePresenter
{
	/** @persistent */
	public ?string $orderIdent = null;

	protected Payments    $paymentsService;
	protected Orders      $ordersService;
	protected Statuses    $statusesService;
	protected QrGenerator $qrGenerator;
	protected Sellers     $sellers;
	protected LinksHelper $linksHelper;

	public ?Order $order = null;

	protected array $includeTemplates = [];

	public function __construct(
		Orders      $orders,
		Statuses    $statuses,
		Payments    $payments,
		QrGenerator $qrGenerator,
		Sellers     $sellers,
		LinksHelper $linksHelper
	)
	{
		$this->ordersService   = $orders;
		$this->statusesService = $statuses;
		$this->paymentsService = $payments;
		$this->qrGenerator     = $qrGenerator;
		$this->sellers         = $sellers;
		$this->linksHelper     = $linksHelper;
	}

	protected function startup(): void
	{
		parent::startup();

		$this['meta']->setMeta('robots', 'noindex, nofollow');
	}

	/*******************************************************************************************************************
	 * ============================== Actions
	 */

	public function actionPay(string $orderIdent): void
	{
		if ($this->getParameter('orderIdent')) {
			$this->order = $this->ordersService->getByIdent($orderIdent);
		}

		if (!$this->order) {
			$this->error();
		}

		$cannotPayWithCard = false;

		if (!in_array($this->order->getPaymentIdent(), ['card', 'essox'])) {
			$cannotPayWithCard = true;
		}

		if ($cannotPayWithCard) {
			$this->template->bigTitle = 'eshopOrdersFront.errors.error';
			$this->template->messages = [
				'eshopOrdersFront.errors.orderCantBePaidWithCard',
			];
		} else {
			$this->eventDispatcher->dispatch(new PresenterTemplateEvent($this->template, $this), 'eshopOrders.paymentPayAction');
			$this->template->includeTemplates = $this->includeTemplates;
		}
	}

	public function actionPaymentFinished(): void
	{
		$orderIdent = $this->getParameter('orderIdent');

		if ($orderIdent) {
			// Podle ident
			$this->order = $this->ordersService->getByIdent($orderIdent);
		} else {
			// Podle id
			$orderId = $this->getParameter('orderId');

			if ($orderId) {
				/** @var Order|null $tmpOrder */
				$tmpOrder = $this->ordersService->getEr()->find((int) $orderId);

				$this->order = $tmpOrder;
			} else {
				// Comgate
				$orderIdent = $this->getHttpRequest()->getPost('refId');

				if ($orderIdent) {
					/** @var Order|null $tmpOrder */
					$tmpOrder = $this->ordersService->getEr()->find((int) $orderIdent);

					$this->order = $tmpOrder;
				}
			}
		}

		$reason = $this->getParameter('reason');

		if ($this->order) {
			$this->template->order = $this->order;
		}

		$this->template->reason = $reason;

		$this->eventDispatcher->dispatch(new PresenterTemplateEvent($this->template, $this), 'eshopOrders.paymentFinishedRender');
		$this->template->includeTemplates = $this->includeTemplates;

		if ($this->order && $this->order->lang !== $this->translator->getLocale()) {
			$links = $this->linksHelper->getOtherLangLinks($this->getActiveNavigation(), $this->getParameters(), $this->getName());
			if (array_key_exists($this->order->lang, $links)) {
				$url = new Url($links[$this->order->lang]['link']);
				foreach ([
							 'reason'  => $reason,
							 'orderId' => $this->order->getId(),
							 'transId' => $this->getParameter('transId'),
						 ] as $name => $value) {
					if ($value) {
						$url->setQueryParameter($name, $value);
					}
				}
				$this->redirectUrl((string) $url);
			}
		}
	}

	public function actionQrCode(string $orderIdent, ?string $size = null): void
	{
		$order = $this->ordersService->getByIdent($orderIdent ?: '0');

		if (!$order) {
			throw new BadRequestException();
		}

		$seller = $this->sellers->getSellerForSite($order->site->getIdent());
		if ($seller) {
			$qrPayment = $this->qrGenerator->getQrCodeByOrderAndSeller($order, $seller);

			if ($qrPayment) {
				$response = $this->getHttpResponse();
				$response->setContentType('image/png');
				$rawString = $qrPayment->getQrCode()->getRawString();

				$img = Image::fromString($rawString);

				if ($size) {
					[$width, $height] = explode('x', $size);
					$img->resize((int) $width, (int) $height, Image::FIT);
				}
				FileSystem::createDir(TMP_DIR . '/qrCode/');
				$file = TMP_DIR . '/qrCode/' . $orderIdent . '.png';
				$img->save($file);

				$response = new FileResponse($file, $orderIdent, 'image/png', false);

				$this->sendResponse($response);
			}
		}

		throw new BadRequestException();
	}

	/*******************************************************************************************************************
	 * ============================== Renders
	 */

	public function renderPaymentFinished(): void
	{
		//flash zpravy zobrazime v textu, ne jako prekryvne
		$this->template->messages     = $this->getTemplate()->flashes;
		$this->getTemplate()->flashes = [];
	}

	/*******************************************************************************************************************
	 * ============================== Components
	 */

	public function addIncludeTemplate(string $file): self
	{
		$this->includeTemplates[] = $file;

		return $this;
	}
}
