<?php declare(strict_types = 1);

namespace EshopOrders\Model\Entities;

use Core\Model\Entities\TId;
use DateTimeInterface;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Nette\Utils\DateTime;
use Users\Model\Entities\User;

/**
 * @ORM\Table("eshop_orders__cart", indexes={@ORM\Index(name="ident", columns={"ident"})})
 * @ORM\Entity
 */
class Cart
{
	use TId;

	/**
	 * @ORM\Column(name="ident", type="string", nullable=true)
	 */
	public ?string $ident = null;

	/**
	 * @ORM\Column(name="site_ident", type="string", nullable=true)
	 */
	public ?string $siteIdent = null;

	/**
	 * @var ArrayCollection<CartItem>
	 * @ORM\OneToMany(targetEntity="CartItem", mappedBy="cart", indexBy="ident")
	 */
	public Collection $cartItems;

	/**
	 * @ORM\Column(name="order_gift_id", type="integer", nullable=true, options={"unsigned": true})
	 */
	public ?int $orderGiftId = null;

	/**
	 * @var DateTimeInterface
	 * @ORM\Column(name="created", type="datetime", nullable=false, options={"default":"CURRENT_TIMESTAMP"})
	 */
	private $created;

	/**
	 * @ORM\Column(name="more_data", type="array", nullable=true)
	 */
	protected ?array $moreData = [];

	/**
	 * @ORM\Column(name="last_activity", type="datetime", nullable=true, options={"default":"CURRENT_TIMESTAMP"})
	 */
	public ?DateTimeInterface $lastActivity = null;

	/**
	 * @ORM\ManyToOne(targetEntity="Users\Model\Entities\User")
	 * @ORM\JoinColumn(name="user_id", referencedColumnName="id", nullable=true, onDelete="CASCADE")
	 */
	public ?User $user = null;

	/**
	 * @ORM\Column(name="`name`", type="string", nullable=true)
	 */
	public ?string $name = null;

	/**
	 * @ORM\Column(name="`type`", type="string", nullable=false, options={"default": "default"})
	 */
	public string $type = 'default';

	public function __construct(string $ident, string $siteIdent)
	{
		$this->ident        = $ident;
		$this->siteIdent    = $siteIdent;
		$this->cartItems    = new ArrayCollection();
		$this->created      = new DateTime();
		$this->lastActivity = new DateTime();
		$this->moreData     = [];
	}

	/******
	 * === Cart Items
	 */

	/**
	 * @return ArrayCollection<CartItem>
	 */
	public function getCartItems() { return $this->cartItems; }

	public function addCartItem(CartItem $itemRaw): void
	{
		$this->cartItems->add($itemRaw);
	}

	public function removeCartItem(int $itemId): void
	{
		$this->cartItems->remove($itemId);
	}

	public function getCreated(): DateTimeInterface { return $this->created; }

	public function getMoreData(): array { return $this->moreData ?? []; }

	/** @return mixed */
	public function getMoreDataValue(string $key)
	{
		return $this->moreData[$key] ?? null;
	}

	public function setMoreData(array $data): self
	{
		$this->moreData = $data;

		return $this;
	}

	/** @param mixed $value */
	public function setMoreDataValue(string $key, $value): self
	{
		if (!is_array($this->moreData)) {
			$this->moreData = [];
		}

		$this->moreData[$key] = $value;

		return $this;
	}
}

