<?php declare(strict_types = 1);

namespace EshopOrders\Model\Entities;

use Core\Model\Entities\Country;
use Core\Model\Entities\TId;
use Doctrine\ORM\Mapping as ORM;

/**
 * @ORM\Table("eshop_orders__order_address")
 * @ORM\Entity
 */
class OrderAddress
{
	use TId;

	public const ADDRESS_DELIVERY = 'delivery';
	public const ADDRESS_INVOICE  = 'invoice';

	/**
	 * @ORM\Column(name="type", type="string", nullable=false)
	 * hodnoty: dodaci, fakturacni, pro postu = pobocka/balikovna, adresa obdarovaneho, a podobne
	 * od kazdeho typu muze mit objednavka maximalne jednu adresu
	 * objednavka musi mit minimalne dodaci adresu, ostatni jsou nepovinne
	 */
	protected string $type;

	/**
	 * @ORM\Column(name="first_name", type="string", length=255, nullable=false)
	 */
	protected string $firstName;

	/**
	 * @ORM\Column(name="last_name", type="string", length=255, nullable=true)
	 */
	protected ?string $lastName = null;

	/**
	 * @ORM\Column(name="email", type="string", length=255, nullable=true)
	 */
	protected ?string $email = null;

	/**
	 * @ORM\Column(name="phone", type="string", length=255, nullable=true)
	 */
	protected ?string $phone = null;

	/**
	 * @ORM\Column(name="company", type="string", nullable=true)
	 */
	protected ?string $company = null;

	/**
	 * @ORM\Column(name="street", type="string", nullable=true)
	 */
	protected ?string $street = null;

	/**
	 * @ORM\Column(name="city", type="string", nullable=true)
	 */
	protected ?string $city = null;

	/**
	 * @ORM\Column(name="postal", type="string", nullable=true)
	 */
	protected ?string $postal = null;

	/**
	 * @ORM\ManyToOne(targetEntity="Core\Model\Entities\Country")
	 * @ORM\JoinColumn(name="country_id", referencedColumnName="id", onDelete="SET NULL")
	 */
	protected ?Country $country = null;

	/**
	 * @ORM\Column(name="id_number", type="string", nullable=true)
	 */
	protected ?string $idNumber = null;

	/**
	 * @ORM\Column(name="vat_number", type="string", nullable=true)
	 */
	protected ?string $vatNumber = null;

	/**
	 * @ORM\Column(name="validated_id_number", type="smallint", length=1, options={"default": 0})
	 */
	public int $validatedIdNumber = 0;

	/**
	 * @ORM\Column(name="validated_vat_number", type="smallint", length=1, options={"default": 0})
	 */
	public int $validatedVatNumber = 0;

	/**
	 * @ORM\Column(name="id_vat_number", type="string", length=255, nullable=true)
	 */
	public ?string $idVatNumber = null;

	/* property $order nenastavuji - kvuli Doctrine by se muselo odlisit:
	 *		- "objednavka, pro kterou je toto dodaci adresa"
	 *		- "objednavka, pro kterou je toto fakturacni adresa"
	 *		- "objednavka, pro kterou je toto adresa posty"
	 *		- ....
	 *	prestoze by byla vyplnena vzdy jen jedna z techto adres
	 */

	public function __construct(string $type)
	{
		$this->type = $type;
	}

	public function getType(): string { return $this->type; }

	public function getFirstName(): string { return $this->firstName; }

	public function getLastName(): ?string { return $this->lastName; }

	public function getName(): string { return trim($this->getFirstName() . ' ' . $this->getLastName()); }

	public function getCompany(): ?string { return $this->company; }

	public function getStreet(): ?string { return $this->street; }

	public function getCity(): ?string { return $this->city; }

	public function getPostal(): ?string { return $this->postal; }

	public function setFirstName(string $firstName): self
	{
		$this->firstName = $firstName;

		return $this;
	}

	public function setLastName(?string $lastName): self
	{
		$this->lastName = $lastName;

		return $this;
	}

	public function setCompany(?string $company): self
	{
		$this->company = $company;

		return $this;
	}

	public function setStreet(?string $street): self
	{
		$this->street = $street;

		return $this;
	}

	public function setCity(?string $city): self
	{
		$this->city = $city;

		return $this;
	}

	public function setPostal(?string $postal): self
	{
		$this->postal = $postal;

		return $this;
	}

	public function getEmail(): ?string { return $this->email; }

	public function setEmail(?string $email): OrderAddress
	{
		$this->email = $email;

		return $this;
	}

	public function getPhone(): ?string { return $this->phone; }

	public function setPhone(?string $phone): OrderAddress
	{
		$this->phone = $phone;

		return $this;
	}

	public function getCountry(): ?Country { return $this->country; }

	public function setCountry(?Country $country): OrderAddress
	{
		$this->country = $country;

		return $this;
	}

	public function getIdNumber(): ?string { return $this->idNumber; }

	public function setIdNumber(?string $idNumber): OrderAddress
	{
		$this->idNumber = $idNumber;

		return $this;
	}

	public function getVatNumber(): ?string { return $this->vatNumber; }

	public function setVatNumber(?string $vatNumber): OrderAddress
	{
		$this->vatNumber = $vatNumber;

		return $this;
	}
}
