<?php declare(strict_types = 1);

namespace EshopOrders\Model\Entities;

use Core\Model\Entities\TId;
use Core\Model\Entities\TTranslateListener;
use Core\Model\Helpers\Strings;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use EshopCatalog\Model\Entities\Manufacturer;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\Model\Entities\Supplier;
use EshopOrders\Model\Entities\Attributes\IsReadyToDelivery;
use EshopOrders\Model\PriceCalculator\PriceCalculatorItem;
use Nette\Utils\Html;

/**
 * @ORM\Table("eshop_orders__order_item")
 * @ORM\Entity
 * @ORM\EntityListeners({"Core\Model\Entities\TranslateListener"})
 */
class OrderItem implements IOrderItem
{
	use TId;
	use TTranslateListener;
	use IsReadyToDelivery;

	/**
	 * @ORM\ManyToOne(targetEntity="EshopCatalog\Model\Entities\Product")
	 * @ORM\JoinColumn(name="product_id", referencedColumnName="id", onDelete="SET NULL")
	 */
	protected ?Product $product;

	/**
	 * @ORM\Column(name="code1", type="string", length=60, nullable=true)
	 */
	protected ?string $code1 = null;

	/**
	 * @ORM\Column(name="quantity", type="smallint")
	 */
	protected int $quantity = 1;

	/**
	 * @var float|string
	 * @ORM\Column(name="price", type="decimal", precision=10, scale=2)
	 */
	protected $price;

	/**
	 * @ORM\Column(name="vat_rate", type="smallint", nullable=true)
	 */
	public ?int $vatRate = null;

	/**
	 * @var double|string
	 * @ORM\Column(name="recycling_fee", type="decimal", precision=10, scale=2, nullable=true)
	 */
	public $recyclingFee;

	/**
	 * @ORM\ManyToOne(targetEntity="Order", inversedBy="orderItems")
	 * @ORM\JoinColumn(name="order_id", referencedColumnName="id", onDelete="CASCADE")
	 */
	public Order $order;

	/**
	 * @var ArrayCollection<OrderItemTexts>
	 * @ORM\OneToMany(targetEntity="OrderItemTexts", mappedBy="id", indexBy="lang", cascade={"all"})
	 */
	public Collection $orderItemTexts;

	/**
	 * @var ArrayCollection<OrderItemGift>
	 * @ORM\OneToMany(targetEntity="OrderItemGift", mappedBy="orderItem")
	 */
	protected Collection $gifts;

	/**
	 * @var Collection<OrderItemSale>
	 * @ORM\OneToMany(targetEntity="OrderItemSale", mappedBy="orderItem")
	 */
	public Collection $sales;

	/**
	 * @ORM\Column(name="more_data", type="array", nullable=true)
	 */
	protected ?array $moreData = [];

	/**
	 * @var ArrayCollection<self>
	 * @ORM\OneToMany(targetEntity="OrderItem", mappedBy="parent")
	 */
	protected Collection $children;

	/**
	 * @ORM\ManyToOne(targetEntity="OrderItem", inversedBy="children")
	 * @ORM\JoinColumn(name="parent_id", referencedColumnName="id", onDelete="CASCADE")
	 */
	protected ?OrderItem $parent = null;

	/**
	 * @var ArrayCollection<OrderItemReview>
	 * @ORM\OneToMany(targetEntity="OrderItemReview", mappedBy="orderItem")
	 */
	public Collection $orderItemReview;

	protected ?int $defaultQuantity = null;

	public ?PriceCalculatorItem $priceCalculatorItem         = null;
	public ?PriceCalculatorItem $priceCalculatorItemCurrency = null;

	public function __construct(?Product $product, Order $order)
	{
		$this->product         = $product;
		$this->order           = $order;
		$this->quantity        = 1;
		$this->orderItemTexts  = new ArrayCollection();
		$this->moreData        = [];
		$this->gifts           = new ArrayCollection();
		$this->children        = new ArrayCollection();
		$this->orderItemReview = new ArrayCollection();
		$this->sales           = new ArrayCollection();
	}

	/*****
	 * === Quantity
	 */

	public function setQuantity(int $quantity): void
	{
		$this->quantity = $quantity && $quantity > 0 ? $quantity : null;
	}

	public function getQuantity(): int { return $this->quantity; }

	/*******
	 * === Product
	 */

	public function getProduct(): ?Product { return $this->product; }

	public function getProductId(): ?int { return $this->product ? $this->product->getId() : null; }

	public function getManufacturer(): ?Manufacturer { return $this->getProduct() ? $this->getProduct()->getManufacturer() : null; }

	public function getProductNameWithoutManufacturer(): string
	{
		$manu = $this->getManufacturer();
		$text = $this->getOrderItemText();

		if (!$text || !$manu) {
			return $text->getName();
		}

		return Strings::removeFromStart($text->getName(), $manu->name);
	}

	public function getCode1(): ?string { return $this->code1; }

	public function setCode1(?string $code1): OrderItem
	{
		$this->code1 = $code1;

		return $this;
	}

	/*******
	 * === Order
	 */

	public function getOrder(): Order { return $this->order; }

	public function setOrder(Order $order): self
	{
		$this->order = $order;

		return $this;
	}

	/*******
	 * === Price
	 */
	public function getPriceRaw(bool $ignoreZeroVat = false): float
	{
		if (!$ignoreZeroVat && $this->getOrder()->isZeroVat()) {
			return round((float) $this->price / (1 + ($this->vatRate / 100)), 2);
		}

		return (float) $this->price;
	}

	public function getPrice(bool $useCurrency = false): float
	{
		$this->order->initPriceCalculator();

		if ($this->order->isZeroVat()) {
			return $this->getPriceWithoutVat($useCurrency);;
		}

		return $useCurrency
			? $this->priceCalculatorItemCurrency->getPriceWithVat()->getAmount()->toFloat()
			: $this->priceCalculatorItem->getPriceWithVat()->getAmount()->toFloat();
	}

	public function getPriceTotal(bool $useCurrency = false): float
	{
		$this->order->initPriceCalculator();

		if ($this->order->isZeroVat()) {
			return $this->getPriceTotalWithoutVat($useCurrency);
		}

		return $useCurrency
			? $this->priceCalculatorItemCurrency->getTotalWithVat()->getAmount()->toFloat()
			: $this->priceCalculatorItem->getTotalWithVat()->getAmount()->toFloat();
	}

	public function getPriceWithoutVat(bool $useCurrency = false): float
	{
		$this->order->initPriceCalculator();

		return $useCurrency
			? $this->priceCalculatorItemCurrency->getPriceWithoutVat()->getAmount()->toFloat()
			: $this->priceCalculatorItem->getPriceWithoutVat()->getAmount()->toFloat();
	}

	public function getPriceTotalWithoutVat(bool $useCurrency = false): float
	{
		$this->order->initPriceCalculator();

		return $useCurrency
			? $this->priceCalculatorItemCurrency->getTotalWithoutVat()->getAmount()->toFloat()
			: $this->priceCalculatorItem->getTotalWithoutVat()->getAmount()->toFloat();
	}

	/** @param float $price */
	public function setPrice($price): self
	{
		$this->price = (string) Strings::formatEntityDecimal($price);

		return $this;
	}

	/*******
	 * === VatRate
	 */

	public function getVatRate(): int
	{
		if ($this->getOrder()->isZeroVat()) {
			return 0;
		}

		return $this->vatRate ?? 21;
	}

	public function setVatRate(int $vatRate): OrderItem
	{
		$this->vatRate = $vatRate;

		return $this;
	}

	/*******
	 * === OrderItemText
	 */

	public function addOrderItemText(string $lang): OrderItemTexts
	{
		$text = new OrderItemTexts($this, $lang);
		$this->orderItemTexts->set($lang, $text);

		return $text;
	}

	public function setOrderItemText(OrderItemTexts $orderItemTexts): void
	{
		$this->orderItemTexts->set($orderItemTexts->getLang(), $orderItemTexts);
	}

	/**
	 * @param string|null $lang
	 *
	 * @return OrderItemTexts|null
	 */
	public function getOrderItemText(?string $lang = null): ?OrderItemTexts
	{
		return $this->orderItemTexts->get($lang ?: $this->locale) ?? $this->orderItemTexts->first();
	}


	/*************************************
	 * == MoreData
	 */

	/**
	 * @param mixed|null $default
	 *
	 * @return mixed|null
	 */
	public function getMoreDataValue(string $key, $default = null)
	{
		return $this->moreData[$key] ?? $default;
	}

	public function getMoreData(): array { return $this->moreData ?: []; }

	/**
	 * @param mixed $value
	 */
	public function setMoreDataValue(string $key, $value): self
	{
		if (!is_array($this->moreData)) {
			$this->moreData = [];
		}

		$this->moreData[$key] = $value;

		return $this;
	}

	public function setMoreData(array $data): self
	{
		$this->moreData = $data;

		return $this;
	}

	public function setDefaultQuantity(int $quantity): void
	{
		$this->defaultQuantity = $quantity;
	}

	/*************************************
	 * == Gifts
	 */

	/** @return ArrayCollection<OrderItemGift> */
	public function getGifts() { return $this->gifts; }

	public function addGift(OrderItemGift $gift): self
	{
		$this->gifts->add($gift);

		return $this;
	}

	/*************************************
	 * == Children
	 */

	/** @return ArrayCollection<self> */
	public function getChildren() { return $this->children; }

	public function getParent(): ?OrderItem { return $this->parent; }

	public function setParent(OrderItem $parent): void
	{
		$this->parent = $parent;
		$parent->getChildren()->add($this);
	}

	/*************************************
	 * == Recycling Fee
	 */
	public function getRecyclingFeeWithoutVat(): float
	{
		return (float) $this->recyclingFee;
	}

	public function getTotalRecyclingFeeWithoutVat(): float
	{
		return $this->getQuantity() * $this->getRecyclingFeeWithoutVat();
	}

	public function getSuppliersLinks(): ?Html
	{
		$product = $this->getProduct();
		if (!$product) {
			return null;
		}

		$wrap = Html::el();
		foreach ($product->getSuppliers()->toArray() as $supp) {
			$supp = $supp->getSupplier();
			/** @var Supplier $supp */
			$div = Html::el('div');
			if ($supp->website) {
				$div->addHtml(Html::el('a', [
					'href'   => $supp->website,
					'target' => '_blank',
				])->setText($supp->name));
			} else {
				$div->setText($supp->name);
			}

			$wrap->addHtml($div);
		}

		return $wrap;
	}

	public function getUploadedFilesDir(): string
	{
		return self::getBaseUploadedFilesDir($this->order->getId(), $this->id);
	}

	public function getUploadedFiles(): array
	{
		return self::getBaseUploadedFiles($this->getUploadedFilesDir());
	}

	public static function getBaseUploadedFilesDir(int $orderId, int $itemId): string
	{
		return UPLOADS_DIR . DS . 'eshoporders-order-items-uploads' . DS . $orderId . DS . $itemId;
	}

	public static function getBaseUploadedFiles(string $dir): array
	{
		$files = [];

		foreach (glob($dir . DS . '*') ?: [] as $file) {
			$files[] = str_replace(WWW_DIR, '', $file);
		}

		return $files;
	}

	public function getDefaultQuantity(): int { return $this->defaultQuantity ?: 0; }
}
