<?php declare(strict_types = 1);

namespace EshopOrders\Model\Entities;

use Core\Model\Entities\TId;
use Core\Model\Helpers\Strings;
use Doctrine\ORM\Mapping as ORM;
use EshopOrders\Model\PriceCalculator\PriceCalculatorItem;

/** podobne jako order_item, ale samostatna tabulka
 * @ORM\Table("eshop_orders__order_payment")
 * @ORM\Entity
 */
class OrderPayment
{
	use TId;

	/**
	 * @ORM\ManyToOne(targetEntity="Payment")
	 * @ORM\JoinColumn(name="payment_id", referencedColumnName="id", onDelete="SET NULL")
	 */
	public ?Payment $payment = null;

	/**
	 * @ORM\Column(name="name", type="string")
	 */
	protected ?string $name = null;

	/**
	 * @var float|string
	 * @ORM\Column(name="price", type="decimal", precision=10, scale=2)
	 */
	protected $price;

	/**
	 * @ORM\OneToOne(targetEntity="Order", inversedBy="payment")
	 * @ORM\JoinColumn(name="order_id", referencedColumnName="id", onDelete="CASCADE")
	 */
	public ?Order $order = null;

	/**
	 * @ORM\Column(name="vat_rate", type="smallint", nullable=true)
	 */
	protected ?int $vatRate = null;

	public ?PriceCalculatorItem $priceCalculatorItem         = null;
	public ?PriceCalculatorItem $priceCalculatorItemCurrency = null;

	public function __construct(?Payment $payment, Order $order)
	{
		$this->payment = $payment;
		$this->name    = $payment ? $payment->getName() : '';
		$this->order   = $order;

		if ($payment) {
			$this->price = $payment->getPrice();
		}
	}

	public function getIdent(): ?string { return $this->getPayment() ? $this->getPayment()->getIdent() : null; }

	public function getPaymentId(): ?int { return $this->payment ? $this->payment->getId() : null; }

	public function getPayment(): ?Payment { return $this->payment; }

	public function setPayment(Payment $payment): self
	{
		$this->payment = $payment;

		return $this;
	}

	public function getName(): string { return $this->name; }

	public function setName(string $name): self
	{
		$this->name = $name;

		return $this;
	}

	public function getPriceRaw(bool $ignoreZeroVat = false): float
	{
		if (!$ignoreZeroVat && $this->getOrder() && $this->getOrder()->isZeroVat()) {
			return round((float) $this->price / (1 + ($this->vatRate / 100)), 2);
		}

		return (float) $this->price;
	}

	public function getPrice(bool $useCurrency = false): float
	{
		$this->order->initPriceCalculator();

		return $useCurrency
			? $this->priceCalculatorItemCurrency->getTotalWithVat()->getAmount()->toFloat()
			: $this->priceCalculatorItem->getTotalWithVat()->getAmount()->toFloat();
	}

	public function getPriceWithoutVat(bool $useCurrency = false): float
	{
		$this->order->initPriceCalculator();

		return $useCurrency
			? $this->priceCalculatorItemCurrency->getTotalWithoutVat()->getAmount()->toFloat()
			: $this->priceCalculatorItem->getTotalWithoutVat()->getAmount()->toFloat();
	}

	public function setPrice(float $price): self
	{
		$this->price = (string) Strings::formatEntityDecimal($price);

		return $this;
	}

	public function getOrder(): ?Order { return $this->order; }

	public function setVatRate(?int $vatRate): void { $this->vatRate = $vatRate; }

	public function getVatRate(): int { return $this->getOrder()->isZeroVat() ? 0 : ($this->vatRate ?: 21); }
}
