<?php declare(strict_types = 1);

namespace EshopOrders\Model\Entities;

use Core\Model\Entities\TId;
use Core\Model\Helpers\Strings;
use Doctrine\ORM\Mapping as ORM;
use EshopOrders\Model\PriceCalculator\PriceCalculatorItem;

/** podobne jako order_item, ale samostatna tabulka
 * @ORM\Table("eshop_orders__order_spedition")
 * @ORM\Entity
 */
class OrderSpedition
{
	use TId;

	/**
	 * @ORM\ManyToOne(targetEntity="Spedition")
	 * @ORM\JoinColumn(name="spedition_id", referencedColumnName="id", onDelete="SET NULL")
	 */
	public ?Spedition $spedition = null;

	/**
	 * @ORM\Column(name="name", type="string")
	 */
	protected ?string $name = null;

	/**
	 * @var float|string
	 * @ORM\Column(name="price", type="decimal", precision=10, scale=2)
	 */
	protected $price;

	/**
	 * @ORM\OneToOne(targetEntity="Order", inversedBy="spedition")
	 * @ORM\JoinColumn(name="order_id", referencedColumnName="id", onDelete="CASCADE")
	 */
	public ?Order $order = null;

	/**
	 * @ORM\Column(name="vat_rate", type="smallint", nullable=true)
	 */
	public ?int $vatRate = null;

	/**
	 * @ORM\Column(name="tracking_number", type="string", length=255, nullable=true)
	 */
	public ?string $trackingNumber = null;

	/**
	 * @ORM\Column(name="tracking_url", type="string", length=255, nullable=true)
	 */
	public ?string $trackingUrl = null;

	/**
	 * @ORM\Column(name="delivery_point_id", type="string", length=255, nullable=true)
	 */
	public ?string $deliveryPointId = null;

	public ?PriceCalculatorItem $priceCalculatorItem         = null;
	public ?PriceCalculatorItem $priceCalculatorItemCurrency = null;

	public function __construct(?Spedition $spedition, Order $order)
	{
		$this->spedition = $spedition;
		$this->name      = $spedition ? $spedition->getName() : '';
		$this->order     = $order;

		if ($spedition) {
			$this->price = $spedition->getPrice();
		}
	}

	public function getIdent(): ?string { return $this->spedition ? $this->spedition->getIdent() : null; }

	public function getSpeditionId(): ?int { return $this->spedition ? $this->spedition->getId() : null; }

	public function getSpedition(): ?Spedition { return $this->spedition; }

	public function setSpedition(Spedition $spedition): self
	{
		$this->spedition = $spedition;

		return $this;
	}

	public function getName(): string { return $this->name; }

	public function setName(string $name): self
	{
		$this->name = $name;

		return $this;
	}

	public function getPriceRaw(bool $ignoreZeroVat = false): float
	{
		if (!$ignoreZeroVat && $this->getOrder() && $this->getOrder()->isZeroVat()) {
			return round((float) $this->price / (1 + ($this->vatRate / 100)), 2);
		}

		return (float) $this->price;
	}

	public function getPrice(bool $useCurrency = false): float
	{
		$this->order->initPriceCalculator();

		return $useCurrency
			? $this->priceCalculatorItemCurrency->getTotalWithVat()->getAmount()->toFloat()
			: $this->priceCalculatorItem->getTotalWithVat()->getAmount()->toFloat();
	}

	public function getPriceWithoutVat(bool $useCurrency = false): float
	{
		$this->order->initPriceCalculator();

		return $useCurrency
			? $this->priceCalculatorItemCurrency->getTotalWithoutVat()->getAmount()->toFloat()
			: $this->priceCalculatorItem->getTotalWithoutVat()->getAmount()->toFloat();
	}

	public function setPrice(float $price): self
	{
		$this->price = (string) Strings::formatEntityDecimal($price);

		return $this;
	}

	public function getOrder(): ?Order { return $this->order; }

	public function getPickupTypeTitle(): ?string
	{
		if ($this->getSpedition()
			&& in_array($this->getSpedition()->getIdent(), ['zasilkovna', 'ulozenka', 'dpdPickup', 'balikomat'])) {
			return $this->getName();
		}

		return null;
	}

	public function setVatRate(?int $vatRate): void { $this->vatRate = $vatRate; }

	public function getVatRate(): int { return $this->getOrder()->isZeroVat() ? 0 : ($this->vatRate ?? 21); }
}
