<?php declare(strict_types = 1);

namespace EshopOrders\Model\Entities;

use Core\Model\Entities\Country;
use Core\Model\Entities\TId;
use Core\Model\Helpers\Strings;
use DateTimeInterface;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use EshopCatalog\Model\Entities\VatRate;
use Gedmo\Mapping\Annotation as Gedmo;

/**
 * @ORM\Table("eshop_orders__spedition")
 * @ORM\Entity
 */
class Spedition
{
	use TId;

	/**
	 * @ORM\Column(name="name", type="string")
	 */
	protected ?string $name = null;

	/**
	 * @ORM\Column(name="ident", type="string")
	 */
	protected ?string $ident = null;

	/**
	 * @ORM\Column(name="code1", type="string", nullable=true)
	 */
	public ?string $code1 = null;

	/**
	 * @ORM\Column(name="code2", type="string", nullable=true)
	 */
	public ?string $code2 = null;

	/**
	 * Vydejni misto
	 *
	 * @ORM\Column(name="is_pickup", type="smallint", nullable=false, options={"default": 0, "unsigned": true})
	 */
	public int $isPickup = 0;

	/**
	 * Osobni odber na prodejne (jen pro zmenu zobrazeni doprav v kosiku)
	 *
	 * @ORM\Column(type="smallint", nullable=false, options={"default": 0, "unsigned": true})
	 */
	public int $isStorePickup = 0;

	/**
	 * @ORM\Column(name="allow_oversized", type="smallint", nullable=false, options={"default": 0, "unsigned": true})
	 */
	public int $allowOversized = 0;

	/**
	 * @ORM\Column(name="show_only_if_oversized_product_in_cart", type="smallint", nullable=false, options={"default": 0, "unsigned": true})
	 */
	public int $showOnlyIfOversizedProductInCart = 0;

	/**
	 * @ORM\Column(name="text", type="string")
	 */
	protected ?string $text = null;

	/**
	 * @ORM\Column(name="is_published", type="smallint")
	 */
	public ?int $isPublished = 0;

	/**
	 * @Gedmo\SortablePosition
	 * @ORM\Column(name="position", type="integer")
	 */
	protected ?int $position = 0;

	/**
	 * @var float|string
	 * @ORM\Column(name="price", type="decimal", precision=10, scale=2)
	 */
	protected $price;

	/**
	 * @ORM\Column(name="free_from", type="integer", nullable=true)
	 */
	protected ?int $freeFrom = null;

	/**
	 * @ORM\Column(name="available_from", type="integer", nullable=true)
	 */
	protected ?int $availableFrom = null;

	/**
	 * @ORM\Column(name="available_to", type="integer", nullable=true)
	 */
	protected ?int $availableTo = null;

	/**
	 * @ORM\Column(name="image", type="string", nullable=true, length=255)
	 */
	public ?string $image = null;

	/**
	 * @ORM\Column(name="zbozi_id", type="string", nullable=true, length=255)
	 */
	public ?string $zboziId = null;

	/**
	 * @ORM\Column(name="heureka_id", type="string", nullable=true, length=255)
	 */
	public ?string $heurekaId = null;

	/**
	 * @ORM\Column(name="google_id", type="string", nullable=true, length=255)
	 */
	public ?string $googleId = null;

	/**
	 * @ORM\Column(name="idealo_id", type="string", nullable=true, length=255)
	 */
	public ?string $idealoId = null;

	/**
	 * @var ArrayCollection<SpeditionPriceLevel>
	 * @ORM\OneToMany(targetEntity="SpeditionPriceLevel", mappedBy="spedition", indexBy="group_id")
	 */
	protected Collection $priceLevels;

	/**
	 * @var ArrayCollection<Customer>
	 * @ORM\ManyToMany(targetEntity="Customer", mappedBy="disabledSpeditions", indexBy="id")
	 */
	protected Collection $disabledForCustomers;

	/**
	 * @ORM\ManyToOne(targetEntity="EshopCatalog\Model\Entities\VatRate")
	 * @ORM\JoinColumn(name="vat_rate", referencedColumnName="id", onDelete="SET NULL")
	 */
	public ?VatRate $vatRate = null;

	/**
	 * @var ArrayCollection<SpeditionWeight>
	 * @ORM\OneToMany(targetEntity="SpeditionWeight", mappedBy="spedition", indexBy="weightFrom")
	 * @ORM\OrderBy({"weightFrom" = "ASC"})
	 */
	public Collection $weights;

	/**
	 * @var float|string
	 * @ORM\Column(name="max_weight", type="decimal", precision=10, scale=2, options={"unsigned": true}, nullable=true)
	 */
	protected $maxWeight = null;

	/**
	 * @ORM\Column(type="string", length=255, nullable=true)
	 */
	public ?string $externalIdent = null;

	/**
	 * @ORM\Column(type="datetime", nullable=true)
	 */
	public ?DateTimeInterface $freeFromDate = null;

	/**
	 * @ORM\Column(type="datetime", nullable=true)
	 */
	public ?DateTimeInterface $freeToDate = null;

	/**
	 * @ORM\ManyToOne(targetEntity="Core\Model\Entities\Country")
	 * @ORM\JoinColumn(name="country_id", referencedColumnName="id", onDelete="SET NULL")
	 */
	public ?Country $country = null;

	/**
	 * @ORM\Column(type="string", length=255, nullable=true)
	 */
	public ?string $frontGroup = null;

	public function __construct()
	{
		$this->isPublished = 1;
		$this->setPosition(-1);
		$this->freeFrom             = 999999;
		$this->priceLevels          = new ArrayCollection();
		$this->disabledForCustomers = new ArrayCollection();
		$this->weights              = new ArrayCollection();
	}

	public function getName(): ?string { return $this->name; }

	public function getText(): ?string { return $this->text; }

	public function getPosition(): int { return $this->position; }

	public function getPrice(): float { return (float) $this->price; }

	public function setName(?string $name): self
	{
		$this->name = $name;

		return $this;
	}

	public function setText(?string $text): self
	{
		$this->text = $text;

		return $this;
	}

	public function setPosition(int $position): self
	{
		$this->position = $position;

		return $this;
	}

	public function setPrice(float $price): self
	{
		$this->price = (string) Strings::formatEntityDecimal($price);

		return $this;
	}

	/*******
	 * === Ident
	 */

	public function getIdent(): string { return $this->ident; }

	public function setIdent(?string $ident): Spedition
	{
		$this->ident = $ident;

		return $this;
	}

	/*******
	 * === FreeFrom
	 */

	public function getFreeFrom(): ?int { return $this->freeFrom; }

	public function setFreeFrom(?int $freeFrom): Spedition
	{
		$this->freeFrom = $freeFrom;

		return $this;
	}

	/*******
	 * === AvailableFrom
	 */

	public function getAvailableFrom(): ?int { return $this->availableFrom; }

	public function setAvailableFrom(int $availableFrom): Spedition
	{
		$this->availableFrom = $availableFrom;

		return $this;
	}

	/*******
	 * === AvailableTo
	 */

	public function getAvailableTo(): ?int { return $this->availableTo; }

	public function setAvailableTo(int $availableTo): Spedition
	{
		$this->availableTo = $availableTo;

		return $this;
	}

	/**
	 * @return ArrayCollection<SpeditionPriceLevel>
	 */
	public function getPriceLevels() { return $this->priceLevels; }

	/**
	 * @param SpeditionPriceLevel[] $priceLevels
	 */
	public function setPriceLevels(array $priceLevels): void
	{
		$this->priceLevels = new ArrayCollection($priceLevels);
	}

	public function setMaxWeight(float $maxWeight): void { $this->maxWeight = $maxWeight > 0 ? (string) Strings::formatEntityDecimal($maxWeight) : null; }

	public function getMaxWeight(): float { return (float) $this->maxWeight; }
}
