<?php declare(strict_types = 1);

namespace EshopOrders\Model;

use Core\Model\Helpers\BaseEntityService;
use Doctrine\DBAL\LockMode;
use Doctrine\ORM\NonUniqueResultException;
use Doctrine\ORM\Query\Expr\Join;
use Doctrine\ORM\QueryBuilder;
use EshopCatalog\Model\Entities\Seller;
use EshopOrders\Model\Entities\Invoice;
use EshopOrders\Model\Entities\InvoiceConfig;
use EshopOrders\Model\Entities\NumericalSeries;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderStatus;
use EshopOrders\Model\Entities\SellerInvoiceConfig;
use EshopOrders\Model\Listeners\InvoiceListener;
use Nette\InvalidArgumentException;
use Nette\Utils\DateTime;
use Nette\Utils\Strings;

class InvoiceConfigRepository extends BaseEntityService
{
	/** @var string */
	protected $entityClass = InvoiceConfig::class;

	protected static function getInvoiceNumber(NumericalSeries $numSeries, string $startNum, DateTime $date = null): string
	{
		$prefix = $numSeries->getRealPrefix($date);
		$serialNumber  = Strings::padLeft($startNum, $numSeries->digitsCount, '0');

		return sprintf('%s%s', $prefix, $serialNumber);
	}

	/**
	 * Vrati fakturu/odd s nejvyssim oznacenim v danem duzp roce ci mesici aroce
	 */
	protected function getLastInvoiceByDuzpYearMonth(int $year, int $month = null, string $type = InvoiceConfig::TYPE_INVOICE): ?Invoice
	{
		$qb = $this->em->getRepository(Invoice::class)->createQueryBuilder('i');
		$qb->join('i.order', 'o', Join::WITH, 'o.invoice = i.id')
		   ->andWhere('o.isCorrectiveTaxDocument = :isCorrectiveTaxDocument')
		   ->setMaxResults(1);

		$format = $month ? '%Y%m' : '%Y';
		if (EshopOrdersConfig::load('invoice.document.separatedDUZPCreated')) {
			$qb->innerJoin('o.orderStatuses', 'os', Join::WITH, 'os.status = :createStatus')
			   ->setParameter('createStatus', OrderStatus::STATUS_CREATED);

			$qb->addSelect("(COALESCE(DATE_FORMAT(i.duzp, '$format'), DATE_FORMAT(os.created, '$format'), DATE_FORMAT(i.createdAt, '$format'))) as hidden duzpYearMonth");
		} else {
			$qb->addSelect("(COALESCE(DATE_FORMAT(i.duzp, '$format'), DATE_FORMAT(i.createdAt, '$format'))) as hidden duzpYearMonth");
		}

		$qb->andHaving('duzpYearMonth = :yearMonth')
		   ->orderBy('i.ident', 'desc')
		   ->setParameter('yearMonth', $month ? sprintf('%04d%02d', $year, $month) : (string) $year)
		   ->setParameter('isCorrectiveTaxDocument', (int) ($type === InvoiceConfig::TYPE_CORRECTIVE));

		/** @var Invoice|null $inv */
		$inv = $qb->getQuery()->getOneOrNullResult();
		if (!$inv) {
			return null;
		}

		return $inv;
	}

	protected function getLastInvoiceInYearMonthByIdent(int $year, ?int $month, NumericalSeries $numericalSeries, string $type = InvoiceConfig::TYPE_INVOICE): ?Invoice
	{
		$identPrefix = $numericalSeries->getRealPrefix(DateTime::fromParts($year, $month ?? 1, 1));
		$identLength = strlen($identPrefix) + $numericalSeries->digitsCount;
		$qb = $this->em->getRepository(Invoice::class)->createQueryBuilder('i');
		$qb->join('i.order', 'o', Join::WITH, 'o.invoice = i.id')
		   ->andWhere('o.isCorrectiveTaxDocument = :isCorrectiveTaxDocument')
		   ->andWhere($qb->expr()->like('i.ident', $qb->expr()->literal($identPrefix . '%')))
		   ->andWhere($qb->expr()->eq($qb->expr()->length('i.ident'), $identLength))
		   ->setParameter('isCorrectiveTaxDocument', (int) ($type === InvoiceConfig::TYPE_CORRECTIVE))
		   ->orderBy('i.ident', 'desc')
		   ->setMaxResults(1);

		return $qb->getQuery()->getOneOrNullResult();
	}

	/**
	 * Algoritmus pracuje dle oznaceni faktury (ident), tudiz se diva zda v budoucich 5ti letech neexistuje
	 * faktura (dle ident) a pokud ne, tak je brano, vse vyjma toto 5ti lete okno jako minulost
	 */
	protected function getLastCreatedInvoiceFromPast(NumericalSeries $numericalSeries, string $type = InvoiceConfig::TYPE_INVOICE): ?Invoice
	{
		$windowYears = 5;
		$now = new DateTime;
		$year = (int) $now->format('Y');

		$identPrefixes = [];
		for ($i = 1; $i < $windowYears + 1; $i++) { // az od nasledujiciho roku
			$identPrefixes[] = $numericalSeries->getRealPrefix(DateTime::fromParts($year + $i, 1, 1));
		}

		if ($numericalSeries->containsPrefixMonthWildcards()) {
			$identPrefixes = [];
			for ($i = 0; $i < $windowYears; $i++) {
				for ($y = 0; $y < 12; $y++) {
					if ($i === 0 && $y <= ((int) $now->format('n'))) { // v aktualnim roce az od nasledujiciho mesice
						continue;
					}
					$identPrefixes[] = $numericalSeries->getRealPrefix(DateTime::fromParts($year + $i, $y + 1, 1));
				}
			}
		}

		$identLength = strlen($identPrefixes[0]) + $numericalSeries->digitsCount;
		$qb = $this->em->getRepository(Invoice::class)->createQueryBuilder('i');
		$qb->join('i.order', 'o', Join::WITH, 'o.invoice = i.id')
		   ->andWhere('o.isCorrectiveTaxDocument = :isCorrectiveTaxDocument')
		   ->andWhere($qb->expr()->eq($qb->expr()->length('i.ident'), $identLength))
		   ->setParameter('isCorrectiveTaxDocument', (int) ($type === InvoiceConfig::TYPE_CORRECTIVE))
		   ->orderBy('i.ident', 'desc')
		   ->setMaxResults(1);

		foreach ($identPrefixes as $identPrefix) {
		   $qb->andWhere($qb->expr()->notLike('i.ident', $qb->expr()->literal($identPrefix . '%')));
		}

		return $qb->getQuery()->getOneOrNullResult();
	}

	public function getQueryBuilder(): QueryBuilder
	{
		return $this->getEr()->createQueryBuilder('ic');
	}

	public function getDueDate(int $sellerId): ?\DateTime
	{
		$now           = new \DateTime;
		$defaultConfig = $this->getConfigBySeller($sellerId);

		if (!$defaultConfig) {
			return null;
		}

		return $now->modify(sprintf('+ %s days', $defaultConfig->maturity));
	}

	public function getNumericalSeries(int $sellerId, string $type): NumericalSeries
	{
		$defaultConfig = $this->getConfigBySeller($sellerId);

		if (!$defaultConfig) {
			throw new InvalidArgumentException('Numerical series for seller not set');
		}

		switch ($type) {
			case InvoiceConfig::TYPE_INVOICE:
				return $defaultConfig->numericalSeries;
			case InvoiceConfig::TYPE_CORRECTIVE:
				return $defaultConfig->correctiveNumericalSeries;
			case InvoiceConfig::TYPE_RECEIPT:
				return $defaultConfig->receiptNumericalSeries;
			default:
				throw new InvalidArgumentException('Value in $type arg is unknown');
		}
	}

	public function generateIdent(Order $order, int $sellerId, string $type = InvoiceConfig::TYPE_INVOICE, DateTime $overrideDuzp = null): ?string
	{
		$now = new DateTime;
		$currentDuzpYear = (int) $now->format('Y');
		$currentDuzpMonth = (int) $now->format('n');
		$duzp = $overrideDuzp ?? ($order->getDUZP() ? DateTime::from($order->getDUZP()) : null);
		$invoiceDuzpYear = $duzp ? ((int) $duzp->format('Y')) : null;
		$invoiceDuzpMonth = $duzp ? ((int) $duzp->format('n')) : null;

		if (!$invoiceDuzpYear || !$invoiceDuzpMonth) {
			return null;
		}

		$nsId = (int) $this->getNumericalSeries($sellerId, $type)->getId();

		/** @var NumericalSeries $numericalSeries */
		$numericalSeries = $this->em->getRepository(NumericalSeries::class)->createQueryBuilder('ns')
			->where('ns.id = :id')
			->setParameter('id', $nsId)
			->setMaxResults(1)
			->getQuery()
			->setLockMode(LockMode::PESSIMISTIC_WRITE)
			->getOneOrNullResult();

		$strMaxOrder = $numericalSeries->getMaxNumber();
		$containsPrefixYearWildcards = $numericalSeries->containsPrefixYearWildcards();
		$containsPrefixMonthWildcards = $numericalSeries->containsPrefixMonthWildcards();
		$startNumber = $numericalSeries->startNumber;

		$getLastDocumentData = function() use ($numericalSeries, $type) {
			switch ($type) {
				case InvoiceConfig::TYPE_INVOICE:
				case InvoiceConfig::TYPE_CORRECTIVE:
					/** @var Invoice|null $lastInvoice */
					$lastInvoice = $this->getLastCreatedInvoiceFromPast($numericalSeries, $type);
					if (!$lastInvoice) {
						return null;
					}

					return ['ident' => $lastInvoice->ident, 'createdAt' => $lastInvoice->createdAt];
				case InvoiceConfig::TYPE_RECEIPT:
					$qb = $this->em->getRepository(Order::class)->createQueryBuilder('o');
					$qb->where('o.receiptIdent IS NOT NULL')
					   ->orderBy('o.receiptIdent', 'desc')
					   ->setMaxResults(1);

					/** @var Order|null $lastReceiptOrder */
					$lastReceiptOrder = $qb->getQuery()->getOneOrNullResult();

					if (!$lastReceiptOrder) {
						return null;
					}

					return ['ident' => $lastReceiptOrder->receiptIdent, 'createdAt' => $lastReceiptOrder->getPaidDate()];
				default:
					throw new InvalidArgumentException('Value in $type arg is unknown');
			}
		};

		$duzpNotThisYearMonth = ($containsPrefixYearWildcards && ($currentDuzpYear !== $invoiceDuzpYear)) || ($containsPrefixMonthWildcards && $currentDuzpMonth !== $invoiceDuzpMonth);

		$resetInvoiceCounter = false;
		if (($containsPrefixMonthWildcards || $containsPrefixYearWildcards) && !$duzpNotThisYearMonth) {

			$lastDocumentData = $getLastDocumentData();

			if ($lastDocumentData !== null) {
				$created = $lastDocumentData['createdAt'];

				// prefix contain month/year wildcards and current month/year is bigger then on the invoice
				if (($containsPrefixMonthWildcards && (((int) $now->format('n')) > ((int) $created->format('n')))) ||
					($containsPrefixYearWildcards && (((int) $now->format('Y')) > ((int) $created->format('Y'))))) {

					$startNumber = 1;

					// v danem roce a mesici jiz muze existovat faktura, reset cisla se tedy provede na aktualni nastaveni counteru teto faktury
					if ($invInThisYearMonth = $this->getLastInvoiceInYearMonthByIdent((int) $now->format('Y'), $containsPrefixMonthWildcards ? ((int) $now->format('n')) : null, $numericalSeries, $type)) {
						$startNumber = $numericalSeries->parseNumber($invInThisYearMonth->ident) + 1;
					}

					InvoiceListener::$enableIncreaseStartNumber = true;
					$resetInvoiceCounter = true;
					$this->resetInvoiceCounter($sellerId, $type, $startNumber);
				}
			}
		}

		// pokud duzp (mesic ci rok) je v budoucnu ci v minulosti -> prideli se cislo mimo aktualni ciselnou radu
		if (!$resetInvoiceCounter && $duzpNotThisYearMonth) {
			InvoiceListener::$enableIncreaseStartNumber = false;
			$invoice = $this->getLastInvoiceInYearMonthByIdent($invoiceDuzpYear, $containsPrefixMonthWildcards ? $invoiceDuzpMonth : null, $numericalSeries, $type);
			$startNumber = 1;
			if (!$invoice) {
				return self::getInvoiceNumber($numericalSeries, (string) $startNumber, $duzp);
			}

			$startNumber = $numericalSeries->parseNumber($invoice->ident) + 1;
			$ident = self::getInvoiceNumber($numericalSeries, (string) $startNumber, $duzp);

			if ((((int) $strMaxOrder) === $startNumber) && Strings::endsWith($ident, $strMaxOrder)) {
				return null;
			}

			return $ident;
		}

		if (((int) $strMaxOrder) === $startNumber) {
			$lastDocumentData = $getLastDocumentData();

			if (Strings::endsWith($lastDocumentData['ident'], $strMaxOrder)) {
				return null;
			}
		}

		InvoiceListener::$enableIncreaseStartNumber = true;
		return self::getInvoiceNumber($numericalSeries, (string) $startNumber);
	}

	private function resetInvoiceCounter(int $sellerId, string $type = InvoiceConfig::TYPE_INVOICE, int $startNumber = 1): void
	{
		$numericalSeries = $this->getNumericalSeries($sellerId, $type);
		$numericalSeries->startNumber = $startNumber;

		$this->em->persist($numericalSeries);
		$this->em->flush($numericalSeries);
	}

	/**
	 * @return SellerInvoiceConfig[]
	 */
	public function getUsed(): array
	{
		$qb = $this->em->createQueryBuilder();
		$qb->select('sic')
			->from(SellerInvoiceConfig::class, 'sic')
			->join('sic.seller', 's');

		return $qb->getQuery()->getResult();
	}

	public function haveAllSellersInvoiceSettingsCreated(): bool
	{
		return $this->em->getRepository(Seller::class)->count([]) === count($this->getUsed());
	}

	/**
	 * @param int $sellerId
	 *
	 * @return InvoiceConfig|null
	 * @throws NonUniqueResultException
	 */
	public function getConfigBySeller(int $sellerId): ?InvoiceConfig
	{
		$qb = $this->em->createQueryBuilder();
		$qb->select('ic')
			->from(InvoiceConfig::class, 'ic')
			->join('ic.sellerInvoiceConfigs', 'sic')
			->join('sic.seller', 's')
			->where('s.id = :id')
			->setParameter('id', $sellerId)
			->setMaxResults(1);

		return $qb->getQuery()->getOneOrNullResult();
	}
}
