<?php declare(strict_types = 1);

namespace EshopOrders\Model;

use Contributte\Translation\Translator;
use Core\Model\Countries;
use Core\Model\Event\Event;
use Core\Model\Event\EventDispatcher;
use Core\Model\Helpers\BaseEntityService;
use Core\Model\Helpers\Strings;
use Core\Model\Images\ImageHelper;
use Core\Model\Notifiers\MailNotifiers\LogNotifier;
use Currency\Model\Config as CurrenciesConfig;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\NonUniqueResultException;
use Doctrine\ORM\NoResultException;
use Doctrine\ORM\Query\Expr\Join;
use Doctrine\ORM\QueryBuilder;
use EshopCatalog\AdminModule\Model\Products;
use EshopCatalog\FrontModule\Model\Dao\BankAccount;
use EshopCatalog\FrontModule\Model\Dao\Seller;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopCatalog\FrontModule\Model\Sellers;
use EshopCatalog\Model\Entities\SellerInSite;
use EshopOrders\AdminModule\Model\Event\InvoiceGenerateEvent;
use EshopOrders\AdminModule\Model\Event\InvoiceRegenerateEvent;
use EshopOrders\Model\Entities\Invoice;
use EshopOrders\Model\Entities\InvoiceConfig;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderItemTexts;
use EshopOrders\Model\Entities\OrderPayment;
use EshopOrders\Model\Entities\OrderStatus;
use EshopOrders\Model\Helpers\OrderHelper;
use EshopOrders\Model\PriceCalculator\PriceCalculator;
use EshopOrders\Model\PriceCalculator\PriceCalculatorDiscount;
use EshopOrders\Model\PriceCalculator\PriceCalculatorItem;
use Exception;
use Nette\Utils\DateTime;
use Nette\Utils\Image;
use Tracy\Debugger;

/**
 * @method Invoice|null get($id)
 */
class Invoices extends BaseEntityService
{
	/** @var string */
	protected $entityClass = Invoice::class;

	protected EventDispatcher         $eventDispatcher;
	protected Translator              $translator;
	protected Sellers                 $sellers;
	protected Products                $products;
	protected ProductsFacade          $productsFacade;
	protected InvoiceConfigRepository $invoiceConfigRepository;
	protected Countries               $countries;

	protected array $duplicatedNewGenerate = [];

	public function __construct(
		EventDispatcher         $eventDispatcher,
		ProductsFacade          $productsFacade,
		Translator              $translator,
		Sellers                 $sellers,
		InvoiceConfigRepository $invoiceConfigRepository,
		Products                $products,
		Countries               $countries
	)
	{
		$this->eventDispatcher         = $eventDispatcher;
		$this->productsFacade          = $productsFacade;
		$this->translator              = $translator;
		$this->sellers                 = $sellers;
		$this->invoiceConfigRepository = $invoiceConfigRepository;
		$this->products                = $products;
		$this->countries               = $countries;
	}

	/**
	 * @return QueryBuilder
	 */
	public function getQueryBuilder(): QueryBuilder
	{
		return $this->getEr()->createQueryBuilder('i')->orderBy('i.ident', 'desc');
	}

	protected function fetchInvoiceByOrder(Order $order): ?Invoice
	{
		$qb = $this->em->createQueryBuilder();
		$qb->select('i, o, s')
			->from(Invoice::class, 'i')
			->innerJoin('i.order', 'o', Join::WITH, 'o.invoice = i.id')
			->innerJoin('i.seller', 's')
			->where('o.id = :orderId')
			->setParameter('orderId', $order->getId());

		return $qb->getQuery()->getOneOrNullResult();
	}

	/**
	 * @param Order  $order
	 * @param bool   $generateIfNotExist
	 * @param string $type
	 *
	 * @return Invoice|null
	 * @throws NonUniqueResultException
	 */
	public function getOneByOrder(Order $order, bool $generateIfNotExist = false, string $type = InvoiceConfig::TYPE_INVOICE, DateTime $overrideDuzp = null): ?Invoice
	{
		/** @var Invoice|null $invoice */
		$invoice = $this->fetchInvoiceByOrder($order);

		if ($invoice) {
			$this->setPriceCalculator($invoice);
		}

		if (!$invoice && $generateIfNotExist) {
			$this->em->beginTransaction();
			$invoice = $this->buildInvoice($order, null, $type, $overrideDuzp);

			$this->em->persist($invoice);
			$this->em->persist($order);
			$this->em->flush($invoice);
			$this->em->flush($order);
			$this->em->commit();

			// TODO docasne logovani
			try {
				$conn = $this->em->getConnection();
				$stmt = $conn->prepare("SELECT count(i.ident) as c FROM eshop_orders__invoice i WHERE i.ident = ?");
				$stmt->execute([$invoice->ident]);

				/** @var int|null $cnt */
				$cnt = $stmt->fetchColumn();
				if (is_numeric($cnt) && $cnt > 1) {
					try {
						throw new Exception($invoice->ident . ' invoice duplicated');
					} catch (Exception $e) {
						if (!file_exists(LOG_DIR . '/invoice_duplicate.log')) {
							$f = fopen(LOG_DIR . '/invoice_duplicate.log', 'wb');

							if ($f) {
								fclose($f);
							}
						}
						Debugger::log($e, 'invoice_duplicate');
						LogNotifier::toDevelopers($e->getMessage(), 'invoice duplicate');
					}
				}
			} catch (Exception $e) {
			}

			if (!$order->getInvoice()) {
				$order->setInvoice($invoice);

				$this->em->persist($order);
				$this->em->flush();
			}

			$this->em->clear();

			/** @var Invoice|null $invoice */
			$invoice = $this->fetchInvoiceByOrder($order);

			if ($invoice) {
				$this->setPriceCalculator($invoice);
			}

			$this->eventDispatcher->dispatch(new InvoiceGenerateEvent($invoice), 'eshopOrders.afterGenerateInvoice');
		}

		return $invoice;
	}

	public function getByDateAndSeller(
		?int               $sellerId = null,
		\DateTimeInterface $from,
		\DateTimeInterface $to,
		?int               $isPaid = null,
		?int               $payment = null,
		?int               $customerGroup = null,
		?string            $byDate = 'default',
		?int               $withoutODD = null
	): array
	{
		$result = [];

		$qb = $this->em->createQueryBuilder()
			->select('i, o, op, sp, oCur, corrTaxDoc, iData, customer, supplier, customerAddr, oAddrDeli, oAddrInv')
			->from(Invoice::class, 'i')
			->innerJoin('i.order', 'o', Join::WITH, 'o.invoice = i.id')
			->innerJoin('i.invoiceData', 'iData')
			->leftJoin('o.addressDelivery', 'oAddrDeli')
			->leftJoin('o.addressInvoice', 'oAddrInv')
			->leftJoin('iData.customer', 'customer')
			->leftJoin('customer.address', 'customerAddr')
			->leftJoin('iData.supplier', 'supplier')
			->leftJoin('iData.payment', 'op')
			->leftJoin('iData.spedition', 'sp')
			->leftJoin('o.currency', 'oCur')
			->leftJoin('o.correctiveTaxDocumentOf', 'corrTaxDoc')
			->setParameters([
				'fromDate' => $from,
				'toDate'   => $to,
			])
			->groupBy('i.id');

		if ($withoutODD === 1) {
			$qb->andWhere('o.isCorrectiveTaxDocument = 0');
		}

		if ($byDate === 'invoiceDUZP') {
			if (EshopOrdersConfig::load('invoice.document.separatedDUZPCreated')) {
				$qb->innerJoin('o.orderStatuses', 'os', Join::WITH, 'os.status = :createStatus')
					->setParameter('createStatus', OrderStatus::STATUS_CREATED);

				$qb->andWhere('CASE WHEN i.duzp IS NULL THEN os.created ELSE i.duzp END >= :fromDate')
					->andWhere('CASE WHEN i.duzp IS NULL THEN os.created ELSE i.duzp END <= :toDate');
			} else {
				$qb->andWhere('CASE WHEN i.duzp IS NULL THEN i.createdAt ELSE i.duzp END >= :fromDate')
					->andWhere('CASE WHEN i.duzp IS NULL THEN i.createdAt ELSE i.duzp END <= :toDate');
			}
		} else {
			$qb->andWhere('i.createdAt >= :fromDate')
				->andWhere('i.createdAt <= :toDate');
		}

		if ($sellerId !== null) {
			$qb->andWhere('i.seller = :sellerId')
				->setParameter('sellerId', $sellerId);
		}

		if ($isPaid !== null) {
			$qb->andWhere('o.isPaid = :isPaid')
				->setParameter('isPaid', $isPaid);
		}

		if ($payment !== null) {
			$qb->innerJoin(OrderPayment::class, 'orderP', Join::WITH, 'orderP.order = o.id AND orderP.payment = :payment')
				->setParameter('payment', $payment);
		}

		if ($customerGroup !== null) {
			if ($customerGroup >= 1) {
				$qb->innerJoin('o.customer', 'orderCustomer', Join::WITH, 'orderCustomer.groupCustomers = :customerGroup')
					->setParameter('customerGroup', $customerGroup);
			} else {
				$qb->leftJoin('o.customer', 'orderCustomer')
					->andWhere('(orderCustomer.groupCustomers IS NULL OR o.customer IS NULL)');
			}
		}

		foreach ($qb->getQuery()->getResult() as $row) {
			/** @var Invoice $row */
			$this->setPriceCalculator($row);

			if ($row->invoiceData && $row->invoiceData->getCustomer() && $row->invoiceData->getCustomer()->getAddress()
				&& !$row->invoiceData->getCustomer()->getAddress()->countryCode) {
				$addr      = $row->invoiceData->getCustomer()->getAddress();
				$countries = array_flip($this->countries->getAllNameColumn());

				if (isset($countries[$addr->country])) {
					$addr->countryCode = $countries[$addr->country];
					$this->em->persist($addr);
					$this->em->flush($addr);
				}
			}

			$result[$row->seller->getId()][] = $row;
		}

		return $result;
	}

	public function getCheckList(
		?int               $sellerId = null,
		\DateTimeInterface $from,
		\DateTimeInterface $to,
		?int               $isPaid = null,
		?int               $payment = null,
		?int               $customerGroup = null,
		string             $byDate = 'default',
		?int               $withoutODD = null
	): array
	{
		$list    = [];
		$allVats = [];
		$items   = [];

		foreach ($this->getByDateAndSeller(
			$sellerId,
			$from,
			$to,
			$isPaid,
			$payment,
			$customerGroup,
			$byDate,
			$withoutODD
		) as $seller => $rows) {
			foreach ($rows as $row) {
				/** @var Invoice $row */
				$this->setPriceCalculator($row);

				$date          = $row->createdAt->format('m/Y');
				$curr          = $row->invoiceData->currency;
				$type          = $row->order->isCorrectiveTaxDocument ? 'corr' : 'inv';
				$key           = $date . '-' . $type . '-' . $curr;
				$otherCurrency = $row->invoiceData->currency !== CurrenciesConfig::load('default', 'CZK');
				$isCanceled    = $row->order->getNewestOrderStatus() == OrderStatus::STATUS_CANCELED;
				$item          = [
					'ident'    => $row->ident,
					'date'     => $row->createdAt->format('d/m/Y'),
					'currency' => $curr,
					'type'     => $type,
					'vats'     => [],
				];

				if (!isset($list[$key])) {
					$list[$key] = [
						'date'           => $date,
						'currency'       => $curr,
						'type'           => $type,
						'count'          => 0,
						'sumBase'        => 0,
						'sumCurr'        => 0,
						'vats'           => [],
						'sumBaseWithVat' => 0,
						'sumCurrWithVat' => 0,
					];
				}

				if (!$isCanceled) {
					$price        = 0;
					$priceWithVat = 0;

					foreach ($row->invoiceData->getVatRates() as $v) {
						$price        += $v['withoutVat'];
						$priceWithVat += $v['total'];
					}

					$list[$key]['count']++;

					$list[$key]['sumCurr']        += $price;
					$item['sumCurr']              = $price;
					$list[$key]['sumCurrWithVat'] += $priceWithVat;
					$item['sumCurrWithVat']       = $priceWithVat;

					$item['subscriber']        = $row->invoiceData->getCustomer()->getName();
					$item['subscriberCompany'] = $row->invoiceData->getCustomer()->company;

					if ($otherCurrency) {
						$price                 = round($price * (float) $row->order->currency->rate);
						$list[$key]['sumBase'] += $price;
						$item['sumBase']       = $price;

						$priceWithVat                 = round($priceWithVat * (float) $row->order->currency->rate);
						$list[$key]['sumBaseWithVat'] += $priceWithVat;
						$item['sumBaseWithVat']       = $priceWithVat;
					} else {
						$list[$key]['sumBase']        += $price;
						$item['sumBase']              = $price;
						$list[$key]['sumBaseWithVat'] += $priceWithVat;
						$item['sumBaseWithVat']       = $priceWithVat;
					}
				}

				foreach ($row->invoiceData->getVatRates() as $k => $v) {
					if (!isset($list[$key]['vats'][$k])) {
						$list[$key]['vats'][$k] = 0;
						$allVats[]              = $k;
					}

					if (!$isCanceled) {
						$price = round($v['total'] - $v['withoutVat'], 2);
						if ($otherCurrency) {
							$price = round($price * (float) $row->order->currency->rate);
						}

						$list[$key]['vats'][$k] += $price;
						$item['vats'][$k]       = $price;
					}
				}

				$items[] = $item;
			}
		}

		$total = [
			'sumBase'        => 0,
			'sumBaseWithVat' => 0,
			'count'          => 0,
			'vats'           => [],
		];
		foreach ($list as $v) {
			$total['sumBase']        += $v['sumBase'];
			$total['sumBaseWithVat'] += $v['sumBaseWithVat'];
			$total['count']          += $v['count'];

			foreach ($v['vats'] as $k => $v) {
				if (!isset($total['vats'][$k]))
					$total['vats'][$k] = 0;

				$total['vats'][$k] += $v;
			}
		}

		ksort($list);

		return [
			'list'    => $list,
			'total'   => $total,
			'allVats' => $allVats,
			'items'   => $items,
		];
	}

	public function getOrdersByDateAndSeller(
		?int               $sellerId = null,
		\DateTimeInterface $from,
		\DateTimeInterface $to,
		?int               $isPaid = null,
		?int               $payment = null,
		?int               $customerGroup = null,
		?int               $withoutODD = null
	): array
	{
		$ids   = [];
		$sites = [];

		$qb = $this->em->createQueryBuilder()->select('IDENTITY(sin.site) as site')
			->from(SellerInSite::class, 'sin');

		if ($sellerId !== null) {
			$qb->where('sin.seller = :seller')
				->setParameter('seller', $sellerId);
		}

		foreach ($qb->getQuery()->getArrayResult() as $row) {
			$sites[] = $row['site'];
		}

		$qbOrders = $this->em->createQueryBuilder()
			->select('o.id')
			->from(Order::class, 'o')
			->innerJoin('o.orderStatuses', 'os', Join::WITH, 'os.status = :status AND os.created >= :fromDate AND os.created<= :toDate')
			->andWhere('o.site IN (:sites)')
			->setParameters([
				'sites'    => $sites,
				'status'   => OrderStatus::STATUS_CREATED,
				'fromDate' => $from,
				'toDate'   => $to,
			]);

		if ($withoutODD === 1) {
			$qbOrders->andWhere('o.isCorrectiveTaxDocument = 0');
		}

		if ($isPaid !== null) {
			$qbOrders->andWhere('o.isPaid = :isPaid')
				->setParameter('isPaid', $isPaid);
		}

		if ($payment !== null) {
			$qbOrders->innerJoin('o.payment', 'orderPayment', Join::WITH, 'orderPayment.payment = :payment')
				->setParameter('payment', $payment);
		}

		if ($customerGroup !== null) {
			if ($customerGroup >= 1) {
				$qbOrders->innerJoin('o.customer', 'orderCustomer', Join::WITH, 'orderCustomer.groupCustomers = :customerGroup')
					->setParameter('customerGroup', $customerGroup);
			} else {
				$qbOrders->leftJoin('o.customer', 'orderCustomer')
					->andWhere('(orderCustomer.groupCustomers IS NULL OR o.customer IS NULL)');
			}
		}

		foreach ($qbOrders->getQuery()->getArrayResult() as $row) {
			$ids[] = $row['id'];
		}

		return $ids;
	}

	/**
	 * @param int $invoiceId
	 *
	 * @return bool
	 */
	public function regenerateInvoice(int $invoiceId): bool
	{
		try {
			$this->em->beginTransaction();

			$qb = $this->em->createQueryBuilder();
			$qb->select('o')
				->from(Order::class, 'o')
				->join('o.invoice', 'i')
				->where('i.id = :invoiceId')
				->setParameter('invoiceId', $invoiceId);

			/** @var Order|null $order */
			$order = $qb->getQuery()->getOneOrNullResult();

			if ($order === null) {
				return false;
			}

			$invoice = $order->getInvoice();

			if ($invoice === null || $invoice->isExternal()) {
				return false;
			}

			$invoice = $this->buildInvoice($order, $invoice);

			$this->eventDispatcher->dispatch(new InvoiceRegenerateEvent($invoice), 'eshopOrders.beforeSaveRegeneratedInvoice');

			$this->em->persist($invoice->invoiceData);
			$this->em->flush();

			$this->em->commit();

			$this->eventDispatcher->dispatch(new InvoiceRegenerateEvent($invoice), 'eshopOrders.afterRegenerateInvoice');

			return true;
		} catch (Exception $exception) {
			Debugger::log($exception);
			$this->em->rollback();

			return false;
		}
	}

	/**
	 * @param array $sellersIds
	 *
	 * @return int
	 * @throws NoResultException
	 * @throws NonUniqueResultException
	 */
	public function getIvoicesCount(array $sellersIds): int
	{
		if (empty($sellersIds)) {
			return 0;
		}

		$qb = $this->em->createQueryBuilder();
		$qb->select('COUNT(i.id)')
			->from(Invoice::class, 'i')
			->join('i.seller', 's')
			->where($qb->expr()->in('s.id', $sellersIds));

		return (int) $qb->getQuery()->getSingleScalarResult();
	}

	protected function buildInvoice(Order $order, ?Invoice $invoice = null, string $type = InvoiceConfig::TYPE_INVOICE, DateTime $overrideDuzp = null): ?Invoice
	{
		$originalType = $type;
		$type         = $type === InvoiceConfig::TYPE_INVOICE_EXTERNAL ? InvoiceConfig::TYPE_INVOICE : $type;

		$isNew = false;
		/** @var Seller $seller */
		$seller = $this->sellers->getSellerForSite($order->site->getIdent());

		if (!$invoice) {
			$sellerId = $seller->id;
			$event    = new Event(['order' => $order]);
			$this->eventDispatcher->dispatch($event, self::class . '::beforeGenerateInvoiceIdent');

			if (isset($event->data['seller']) && $event->data['seller'] !== null) {
				$sellerId = $event->data['seller']->getId();
			}

			$isNew        = true;
			$invoiceIdent = $this->invoiceConfigRepository->generateIdent($order, $sellerId, $type, $overrideDuzp ?? new DateTime);

			if (!$invoiceIdent) {
				return null;
			}

			/** @var \EshopCatalog\Model\Entities\Seller $sellerRef */
			$sellerRef = $this->em->getReference(\EshopCatalog\Model\Entities\Seller::class, $seller->id);

			$invoice        = new Invoice(
				$originalType === InvoiceConfig::TYPE_INVOICE_EXTERNAL
					? null
					: $this->invoiceConfigRepository->getDueDate($sellerId),
				$invoiceIdent,
				$sellerRef
			);
			$invoice->order = $order;
			if ($type !== InvoiceConfig::TYPE_CORRECTIVE) {
				$order->setInvoice($invoice);
			}

			$invoice->duzp = $invoice->createdAt;
		}

		if ($originalType === InvoiceConfig::TYPE_INVOICE_EXTERNAL) {
			$event = new Event(['invoice' => $invoice]);
			$this->eventDispatcher->dispatch($event, self::class . '::afterBuildInvoice');

			return $invoice;
		}

		$invoiceData = $isNew ? null : $invoice->invoiceData;

		if ($invoiceData) {
			$lang = $invoiceData->lang;
		} else if ($order->getParam('invoiceLang')) {
			$lang = $order->getParam('invoiceLang');
		} else if ($order->lang) {
			$lang = $order->lang;
		} else {
			$lang = $this->translator->getLocale();
		}

		// invoice payment
		$invoicePayment = $isNew ? new Invoice\Payment() : $invoiceData->getPayment();
		$orderPayment   = $order->getPayment();

		if ($orderPayment) {
			$invoicePayment->paymentId = (int) $orderPayment->getId();
			$invoicePayment->name      = $orderPayment->getName();
			$invoicePayment->setVatRate($orderPayment->getVatRate());
			$invoicePayment->setPrice(Strings::formatEntityDecimal($orderPayment->getPrice(true)));
		} else {
			$invoicePayment->name = '';
			$invoicePayment->setVatRate(0);
			$invoicePayment->setPrice(0);
		}

		// invoice spedition
		$invoiceSpedition = $isNew ? new Invoice\Spedition() : $invoiceData->getSpedition();
		$orderSpedition   = $order->getSpedition();

		if ($orderSpedition) {
			$invoiceSpedition->speditionId = (string) $orderSpedition->getId();
			$invoiceSpedition->name        = $orderSpedition->getName();
			$invoiceSpedition->setVatRate($orderSpedition->getVatRate());
			$invoiceSpedition->setPrice(Strings::formatEntityDecimal($orderSpedition->getPrice(true)));
		} else {
			$invoiceSpedition->name = '';
			$invoiceSpedition->setVatRate(0);
			$invoiceSpedition->setPrice(0);
		}

		// customer address
		$customerAddress = $isNew ? new Invoice\Address() : $invoiceData->customer->address;
		$addrInv         = $order->getAddressInvoice();

		if ($addrInv) {
			$countryText = null;
			if ($addrInv->getCountry()) {
				$countryText = $addrInv->getCountry()->getText($order->lang)->name;

				if (!$countryText) {
					$countryText = $addrInv->getCountry()->getText()->name;
				}
			}

			$customerAddress->city        = $addrInv->getCity();
			$customerAddress->country     = $countryText;
			$customerAddress->countryCode = $addrInv->getCountry() ? $addrInv->getCountry()->getId() : null;
			$customerAddress->postal      = $addrInv->getPostal();
			$customerAddress->street      = $addrInv->getStreet();
		}

		// customer
		$invoiceCustomer = $isNew ? new Invoice\Customer($customerAddress) : $invoiceData->customer;
		$addrInv         = $order->getAddressInvoice();

		if ($addrInv) {
			$invoiceCustomer->phone              = $addrInv->getPhone();
			$invoiceCustomer->email              = $addrInv->getEmail();
			$invoiceCustomer->company            = $addrInv->getCompany();
			$invoiceCustomer->firstName          = $addrInv->getFirstName();
			$invoiceCustomer->lastName           = $addrInv->getLastName();
			$invoiceCustomer->idNumber           = $addrInv->getIdNumber();
			$invoiceCustomer->vatNumber          = $addrInv->getVatNumber();
			$invoiceCustomer->validatedIdNumber  = $addrInv->validatedIdNumber;
			$invoiceCustomer->validatedVatNumber = $addrInv->validatedVatNumber;
			$invoiceCustomer->idVatNumber        = $addrInv->idVatNumber;
		}

		// bank
		$bank = $isNew ? new Invoice\Bank() : $invoiceData->getSupplier()->getBank();
		// supplier address
		$supplierAddress = $isNew ? new Invoice\Address() : $invoiceData->getSupplier()->getAddress();
		if ($seller && $seller->getBankAccount($invoiceData->currency ?? $order->getCurrencyCode())) {
			/** @var BankAccount $bankAccount */
			$bankAccount       = $seller->getBankAccount($invoiceData->currency ?? $order->getCurrencyCode());
			$bank->bankAccount = $bankAccount->numberPart1;
			$bank->bankCode    = $bankAccount->numberPart2;
			$bank->iban        = $bankAccount->iban;
			$bank->swift       = $bankAccount->swift;
			$bank->note        = $bankAccount->note;

			$supplierAddress->street = $seller->street;
			$supplierAddress->postal = $seller->postal;
			$supplierAddress->city   = $seller->city;

			// supplier
			$supplier             = $isNew ? new Invoice\Supplier($bank, $supplierAddress) : $invoiceData->getSupplier();
			$supplier->email      = $seller->email;
			$supplier->vatNumber  = $seller->dic;
			$supplier->idNumber   = $seller->ic;
			$supplier->name       = $seller->name;
			$supplier->isPayerVat = $seller->dic ? true : false;
		} else {
			$bank->bankAccount = EshopOrdersConfig::load('invoice.bank.bankAccount');
			$bank->bankCode    = EshopOrdersConfig::load('invoice.bank.bankCode');
			$bank->iban        = EshopOrdersConfig::load('invoice.bank.iban');
			$bank->swift       = EshopOrdersConfig::load('invoice.bank.swift');

			$supplierAddress->street = EshopOrdersConfig::load('invoice.supplier.street');
			$supplierAddress->postal = EshopOrdersConfig::load('invoice.supplier.postCode');
			$supplierAddress->city   = EshopOrdersConfig::load('invoice.supplier.city');

			// supplier
			$supplier             = $isNew ? new Invoice\Supplier($bank, $supplierAddress) : $invoiceData->getSupplier();
			$supplier->email      = EshopOrdersConfig::load('invoice.supplier.email');
			$supplier->vatNumber  = EshopOrdersConfig::load('invoice.supplier.taxIdentificationNumber');
			$supplier->idNumber   = EshopOrdersConfig::load('invoice.supplier.companyIdentificationNumber');
			$supplier->name       = EshopOrdersConfig::load('invoice.supplier.name');
			$supplier->isPayerVat = EshopOrdersConfig::load('invoice.isPayerVat');
		}

		if ($isNew) {
			$bank->variableSymbol = $invoiceIdent;

			$invoiceData           = new Invoice\InvoiceData($invoiceCustomer, $supplier, $invoiceSpedition, $invoicePayment, $invoice);
			$invoiceData->lang     = $lang;
			$invoiceData->currency = $order->getCurrencyCode();

			$order->removeParam('invoiceLang');
			$this->em->persist($order);
		}

		$invoiceData->zeroVat = (int) $order->isZeroVat();

		// products
		$invoiceProducts = [];

		if (!$isNew) {
			foreach ($invoiceData->products as $product) {
				$this->em->remove($product);
			}
		}

		foreach ($order->getOrderItems() as $p) {
			$product = $p->getProductId() ? $this->products->get($p->getProductId()) : null;

			if ($product && $product->vatRate) {
				$addrInv = $order->getAddressInvoice();

				if ($addrInv && Strings::lower($addrInv->getCountry() ? $addrInv->getCountry()->getId() : '') === 'cz') {
					$vatRate = OrderHelper::checkCountryVatRate(
						(int) $product->vatRate->rate,
						$addrInv->getCountry()->getId(),
						false,
						$addrInv->getIdNumber(),
						$addrInv->getVatNumber(),
					);

					if ($vatRate !== $p->getVatRate()) {
						$p->setVatRate($vatRate);

						$this->em->persist($p);
						$this->em->flush($p);
					}
				}
			}

			/** @var OrderItemTexts|null $texts */
			$texts = $p->orderItemTexts->get($lang);
			$name  = null;
			if (!$texts && $p->getProduct()) {
				$productTexts = $p->getProduct()->getText($lang);

				if ($productTexts) {
					$name = $productTexts->name;
				}
			}

			if (!$name) {
				if ($texts) {
					$name = $texts->getName();
				} else {
					$name = $p->getOrderItemText($lang)->getName();
				}
			}

			$invoiceProduct            = new Invoice\Product($invoiceData);
			$invoiceProduct->productId = $p->getProductId();
			$invoiceProduct->parentId  = $p->getParent() ? $p->getParent()->getProductId() : null;
			$invoiceProduct->code1     = $p->getCode1();
			$invoiceProduct->name      = $name;
			$invoiceProduct->price     = Strings::formatEntityDecimal($p->getPrice(true));
			$invoiceProduct->quantity  = $p->getQuantity();
			$invoiceProduct->setVatRate($p->getVatRate());
			$invoiceProduct->recyclingFee = Strings::formatEntityDecimal((float) $p->recyclingFee);
			$invoiceProduct->setMoreData([
				'discountDisabled' => $p->getMoreDataValue('discountDisabled', 0),
			]);

			if ($product && $product->getGallery()) {
				$cover = $product->getGallery()->getCoverImage();

				// create base64 image miniature
				if (EshopOrdersConfig::load('invoice.allowImagePreview') && $cover && file_exists(WWW_DIR . $cover->getFilePath())) {
					$invoiceProduct->imageBase64 = ImageHelper::createBase64Miniature(
						WWW_DIR . $cover->getFilePath(),
						EshopOrdersConfig::load('invoice.productMiniature.width'),
						null,
						Image::SHRINK_ONLY);
				}
			}

			$invoiceProducts[] = $invoiceProduct;
		}
		$invoiceData->products = new ArrayCollection($invoiceProducts);

		// discounts
		$invoiceDiscounts = [];

		// remove discounts
		foreach ($invoiceData->discounts as $discount) {
			$this->em->remove($discount);
		}

		$currentPrice = $order->getPriceItems(true);
		foreach ($order->getOrderDiscounts() as $orderDiscount) {
			$amount                 = $orderDiscount->getPrice(true);
			$currentPrice           += $amount;
			$invoiceDiscount        = new Invoice\Discount($invoiceData);
			$invoiceDiscount->price = Strings::formatEntityDecimal($amount);
			$invoiceDiscount->name  = $orderDiscount->getName();
			$invoiceDiscount->type  = $orderDiscount->getType();
			$invoiceDiscount->value = $orderDiscount->getValue();
			$invoiceDiscount->code  = $orderDiscount->getCode();

			$invoiceDiscounts[] = $invoiceDiscount;
		}

		$invoiceData->discounts = new ArrayCollection($invoiceDiscounts);

		if ($isNew) {
			$invoice->invoiceData = $invoiceData;
		}

		$event = new Event(['invoice' => $invoice]);
		$this->eventDispatcher->dispatch($event, self::class . '::afterBuildInvoice');

		return $invoice;
	}

	/**
	 * @return Invoice[]
	 */
	public function getInvoicesOverdue(): array
	{
		$now = (new DateTime)->setTime(23, 59, 59, 59);
		$qb  = $this->getEr()->createQueryBuilder('i');
		$qb->join('i.order', 'o', 'WITH', 'o.invoice = i.id')
			->where('i.dueDate < :date AND o.isPaid = 0')
			->setParameter('date', $now);

		return $qb->getQuery()->getResult();
	}

	public function setPriceCalculator(Invoice $invoice): void
	{
		$order                        = $invoice->order;
		$invoiceData                  = $invoice->invoiceData;
		$invoiceData->priceCalculator = new PriceCalculator($invoiceData->currency, $order && $order->isCorrectiveTaxDocument);

		foreach ($invoiceData->products as $product) {
			$calculatorItem = new PriceCalculatorItem(
				$product->name,
				$product->getPriceRaw(),
				$product->quantity,
				$product->getVatRate(),
				$invoiceData->currency,
				$product->getMoreDataValue('discountDisabled') && (bool) $product->getMoreDataValue('discountDisabled')
			);

			$product->priceCalculatorItem = $calculatorItem;
			$invoiceData->priceCalculator->addItem((string) $product->getId(), $calculatorItem);
		}

		foreach ($invoiceData->discounts as $discount) {
			$calculatorItem = new PriceCalculatorDiscount(
				$discount->type,
				(float) $discount->price,
				$invoiceData->currency
			);

			$discount->priceCalculatorItem = $calculatorItem;
			$invoiceData->priceCalculator->addDiscountCode((string) $discount->getId(), $calculatorItem);
		}

		/** @phpstan-ignore-next-line */
		if ($invoiceData->getSpedition()) {
			$calculatorItem = new PriceCalculatorItem(
				$invoiceData->spedition->name,
				$invoiceData->spedition->getPriceRaw(),
				1,
				$invoiceData->spedition->getVatRate(),
				$invoiceData->currency,
				false
			);

			$invoiceData->spedition->priceCalculatorItem = $calculatorItem;
			$invoiceData->priceCalculator->setSpedition($calculatorItem);
		}

		/** @phpstan-ignore-next-line */
		if ($invoiceData->getPayment()) {
			$calculatorItem = new PriceCalculatorItem(
				$invoiceData->payment->name,
				$invoiceData->payment->getPriceRaw(),
				1,
				$invoiceData->payment->getVatRate(),
				$invoiceData->currency,
				false
			);

			$invoiceData->payment->priceCalculatorItem = $calculatorItem;
			$invoiceData->priceCalculator->setPayment($calculatorItem);
		}
	}
}
