<?php declare(strict_types = 1);

namespace EshopOrders\Model\Listeners;

use Core\Model\Event\Event;
use Core\Model\Event\EventDispatcher;
use Doctrine\ORM\Event\LifecycleEventArgs;
use Doctrine\ORM\Mapping as ORM;
use EshopOrders\Model\Entities\Invoice;
use EshopOrders\Model\InvoiceConfigRepository;

class InvoiceListener
{
	protected InvoiceConfigRepository $invoiceConfigRepository;
	protected EventDispatcher         $eventDispatcher;
	public static bool                $enableIncreaseStartNumber = false;

	public function __construct(
		InvoiceConfigRepository $invoiceConfigRepository,
		EventDispatcher         $eventDispatcher
	)
	{
		$this->invoiceConfigRepository = $invoiceConfigRepository;
		$this->eventDispatcher         = $eventDispatcher;
	}

	/**
	 * @ORM\PrePersist
	 */
	public function prePersist(Invoice $invoice, LifecycleEventArgs $event): void
	{
		if (self::$enableIncreaseStartNumber) {
			$this->persist($invoice, $event);
			self::$enableIncreaseStartNumber = false;
		}
	}

	/**
	 * @ORM\PostUpdate
	 */
	public function postUpdate(Invoice $invoice, LifecycleEventArgs $event): void
	{
		if (self::$enableIncreaseStartNumber) {
			$this->persist($invoice, $event);
			self::$enableIncreaseStartNumber = false;
		}
	}

	public function persist(Invoice $invoice, LifecycleEventArgs $event): void
	{
		$entityManager = $event->getEntityManager();

		$sellerId = $invoice->seller->getId();
		$e = new Event(['order' => $invoice->order]);
		$this->eventDispatcher->dispatch($e, self::class . '::beforeIncreaseStartNumber');

		if (isset($e->data['seller']) && $e->data['seller'] !== null) {
			$sellerId = $e->data['seller']->getId();
		}

		$config = $this->invoiceConfigRepository->getConfigBySeller($sellerId);

		$numericalSeries = $invoice->order->isCorrectiveTaxDocument ? $config->correctiveNumericalSeries : $config->numericalSeries;
		$numericalSeries->increaseStartNumber();

		$entityManager->persist($numericalSeries);
		$entityManager->flush($numericalSeries);
	}

}
