<?php declare(strict_types = 1);

namespace EshopOrders\Model\Navigation;

use Core\Components\Navigation\DaoNavigationItem;
use Core\Model\Helpers\Strings;
use Core\Model\UI\Form\BaseContainer;
use Core\Model\UI\Form\BaseForm;
use EshopOrders\Model\Orders;
use Navigations\Model\Providers\INavigationItem;
use Nette\Utils\ArrayHash;

class Customer extends BaseNavigation implements INavigationItem
{
	protected        $title                   = 'customerOrders';
	protected        $presenter               = 'EshopOrders:Front:Customer';
	protected        $action                  = 'default';
	protected string $actionOrderDetail       = 'orderDetail';
	protected string $actionOrders            = 'orders';
	protected string $actionContactData       = 'contactData';
	protected string $actionAccountSettings   = 'accountSettings';
	protected string $actionEditProfile       = 'editProfile';
	protected string $actionMyProducts        = 'myProducts';
	protected string $actionPurchasedProducts = 'purchasedProducts';

	protected const defaultUrlOrderDetail       = 'order';
	protected const defaultUrlOrders            = 'orders';
	protected const defaultUrlContactData       = 'contact-data';
	protected const defaultUrlAccountSettings   = 'account-settings';
	protected const defaultUrlMyProducts        = 'my-products';
	protected const defaultUrlPurchasedProducts = 'purchased-products';
	protected const URL_EDIT                    = 'edit';

	protected Orders $orders;

	public function __construct(
		Orders $orders
	)
	{
		$this->orders = $orders;
	}

	/**
	 * @param BaseForm $baseForm
	 *
	 * @return BaseContainer
	 */
	public function getFormContainer(BaseForm $baseForm)
	{
		return new BaseContainer();
	}

	public function formSuccess(BaseForm $form, ArrayHash $values)
	{

	}

	/*******************************************************************************************************************
	 * ===========================  Route
	 */

	public function presenterActionCheck($presenter, $action)
	{
		if ($this->presenter === $presenter && $this->action === $action) {
			return 1;
		}

		if ($this->presenter === $presenter && $this->actionOrderDetail === $action) {
			return 2;
		}

		if ($this->presenter === $presenter && $this->actionEditProfile === $action) {
			return 3;
		}

		if (
			$this->presenter === $presenter && in_array($action, [
				$this->actionOrders,
				$this->actionContactData,
				$this->actionAccountSettings,
				$this->actionMyProducts,
				$this->actionPurchasedProducts
			], true)
		) {
			return 5;
		}

		return false;
	}

	public function routerIn($navigation, $urlParams = [])
	{
		if (isset($urlParams['path'])) {

			$params = array_reverse(explode('/', $urlParams['path']));

			if (is_numeric($params[0]) && in_array($params[1], [Strings::webalize($this->t('eshopOrdersFront.customer.urls.orderDetail')), self::defaultUrlOrderDetail], true)) {
				return [
					'presenter' => $this->presenter,
					'action'    => $this->actionOrderDetail,
					'orderId'   => (int) $params[0],
				];
			}

			if (in_array($params[0], [Strings::webalize($this->t('eshopOrdersFront.customer.urls.orders')), self::defaultUrlOrders], true)) {
				return [
					'presenter' => $this->presenter,
					'action'    => $this->actionOrders,
					'id'        => null,
				];
			}

			if (in_array($params[0], [Strings::webalize($this->t('eshopOrdersFront.customer.urls.contactData')), self::defaultUrlContactData], true)) {
				return [
					'presenter' => $this->presenter,
					'action'    => $this->actionContactData,
					'id'        => null,
				];
			}

			if (in_array($params[0], [Strings::webalize($this->t('eshopOrdersFront.customer.urls.accountSettings')), self::defaultUrlAccountSettings], true)) {
				return [
					'presenter' => $this->presenter,
					'action'    => $this->actionAccountSettings,
					'id'        => null,
				];
			}

			if (in_array($params[0], [Strings::webalize($this->t('eshopOrdersFront.customer.urls.myProducts')), self::defaultUrlMyProducts], true)) {
				return [
					'presenter' => $this->presenter,
					'action'    => $this->actionMyProducts,
					'id'        => null,
				];
			}

            if (in_array($params[0], [Strings::webalize($this->t('eshopOrdersFront.customer.urls.purchasedProducts')), self::defaultUrlPurchasedProducts], true)) {
                return [
                    'presenter' => $this->presenter,
                    'action'    => $this->actionPurchasedProducts,
                    'id'        => null,
                ];
            }

			if ($params[0] === self::URL_EDIT) {
				return [
					'presenter' => $this->presenter,
					'action'    => $this->actionEditProfile,
					'id'        => null,
				];
			}
		}

		return [
			'presenter' => $this->presenter,
			'action'    => $this->action,
			'id'        => null,
		];
	}

	public function routerOut(DaoNavigationItem $navigation, &$urlParams = []): ?string
	{
		$urlParts = [];
		if (!$navigation->isHomepage) {
			$urlParts[] = $navigation->alias;
		}

		if (isset($urlParams['orderId'])) {
			$order = $this->orders->get($urlParams['orderId']);
			if ($order) {
				$orderDetailUrl = $this->t('eshopOrdersFront.customer.urls.orderDetail');
				if ($orderDetailUrl === 'customer.urls.orderDetail') {
					$orderDetailUrl = self::defaultUrlOrderDetail;
				}

				$urlParts[] = Strings::webalize($orderDetailUrl);
				$urlParts[] = $order->getId();
			}

			unset($urlParams['orderId']);
		} else if ($urlParams['action'] === $this->actionOrders) {
			$path = $this->t('eshopOrdersFront.customer.urls.orders');
			if ($path === 'customer.urls.orders') {
				$path = self::defaultUrlOrders;
			}

			$urlParts[] = Strings::webalize($path);
		} else if ($urlParams['action'] === $this->actionContactData) {
			$path = $this->t('eshopOrdersFront.customer.urls.contactData');
			if ($path === 'customer.urls.contactData') {
				$path = self::defaultUrlOrders;
			}

			$urlParts[] = Strings::webalize($path);
		} else if ($urlParams['action'] === $this->actionAccountSettings) {
			$path = $this->t('eshopOrdersFront.customer.urls.accountSettings');
			if ($path === 'customer.urls.accountSettings') {
				$path = self::defaultUrlAccountSettings;
			}

			$urlParts[] = Strings::webalize($path);
		} else if ($urlParams['action'] === $this->actionMyProducts) {
			$path = $this->t('eshopOrdersFront.customer.urls.myProducts');
			if ($path === 'customer.urls.myProducts') {
				$path = self::defaultUrlMyProducts;
			}

			$urlParts[] = Strings::webalize($path);
        } else if ($urlParams['action'] === $this->actionPurchasedProducts) {
            $path = $this->t('eshopOrdersFront.customer.urls.purchasedProducts');
            if ($path === 'customer.urls.purchasedProducts') {
                $path = self::defaultUrlPurchasedProducts;
            }

            $urlParts[] = Strings::webalize($path);
		} else if ($urlParams['action'] == $this->actionEditProfile) {
			$urlParts[] = self::URL_EDIT;
		}

		return '/' . implode('/', $urlParts);
	}
}
