<?php declare(strict_types = 1);

namespace EshopOrders\Model;

use Core\Model\Helpers\BaseEntityService;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\Payment;
use EshopOrders\Model\Helpers\EshopOrdersCache;
use Nette\Caching\Cache;
use Tracy\Debugger;

/**
 * @method Payment|null getReference($id)
 * @method Payment|null get($id)
 */
class Payments extends BaseEntityService
{
	protected $entityClass = Payment::class;

	protected EshopOrdersCache $eshopOrdersCache;

	protected ?array $cForSelectOption = null;
	protected ?array $cByIdent         = null;

	public function __construct(
		EshopOrdersCache $eshopOrdersCache
	)
	{
		$this->eshopOrdersCache = $eshopOrdersCache;
	}

	public function setPosition(int $id, int $position): bool
	{
		if ($item = $this->get($id)) {
			$item->setPosition($position);
			$this->em->persist($item);
			$this->em->flush();

			return true;
		}

		return false;
	}

	public function setPublish(int $id, int $state): bool
	{
		if ($item = $this->getReference($id)) {
			$item->isPublished = $state;
			$this->em->persist($item);
			$this->em->flush();

			return true;
		}

		return false;
	}

	/** @return Payment[] */
	public function getAll(): array
	{
		$paymentsQuery = $this->getEr()->createQueryBuilder('p', 'p.id');
		$paymentsQuery->orderBy('p.position');

		return $paymentsQuery->getQuery()->getResult();
	}

	/** @return Payment[] */
	public function getAllPublished(): array
	{
		$paymentsQuery = $this->getEr()->createQueryBuilder('p', 'p.id');
		$paymentsQuery->andWhere('p.isPublished = :isPublished')->setParameter('isPublished', true);
		$paymentsQuery->orderBy('p.position');

		return $paymentsQuery->getQuery()->getResult();
	}

	/**
	 * @param int $cartValue cena objednavky
	 *
	 * @deprecated
	 */
	public function getAllByCartValue($cartValue): Payment
	{
		$paymentsQuery = $this->getEr()->createQueryBuilder('p', 'p.id');
		$paymentsQuery->andWhere('p.isPublished = :isPublished')->setParameter('isPublished', true);
		$paymentsQuery->andWhere('p.availableFrom IS NULL OR p.availableFrom <= :cartValue')->setParameter('cartValue', $cartValue);
		$paymentsQuery->andWhere('p.availableTo IS NULL OR p.availableTo >= :cartValue')->setParameter('cartValue', $cartValue);
		$paymentsQuery->orderBy('p.position');

		return $paymentsQuery->getQuery()->getResult();
	}

	/**
	 * @param string $ident
	 * @param bool   $onlyActive
	 *
	 * @return array{
	 *     name: ?string,
	 *     ident: ?string,
	 *     text: ?string,
	 *     isPublished: int,
	 *     position: int,
	 *     price: float,
	 *     currency: ?string,
	 *     freeFrom: ?int,
	 *     availableFrom: ?int,
	 *     availableTo: ?int,
	 *     image: ?string,
	 *     disableInGoogle: ?int,
	 *     pohodaName: ?string,
	 *     id: int,
	 * }
	 */
	public function getByIdent(string $ident, bool $onlyActive = false): ?array
	{
		if ($this->cByIdent === null) {
			$this->cByIdent = [];

			foreach ($this->getEr()->createQueryBuilder('p')
				         ->getQuery()->getArrayResult() as $row) {
				$row['price'] = (float) $row['price'];

				$this->cByIdent[$row['ident']] = $row;
			}
		}

		$result = $this->cByIdent[$ident] ?? null;

		if ($result && $onlyActive && !$result['isPublished']) {
			return null;
		}

		return $result;
	}

	public function getForSelectOption(): array
	{
		if ($this->cForSelectOption === null) {
			$this->cForSelectOption = [null => ''] + ($this->getEr()->findPairs([], 'name') ?: []);
		}

		return $this->cForSelectOption;
	}

	public function remove($id)
	{
		$result = parent::remove($id);

		if ($result) {
			$this->eshopOrdersCache->getCache()->clean([
				Cache::Tags => ['payment', 'freeFrom'],
			]);
		}

		return $result;
	}

	public function setOrderPayment(
		Order   $order,
		Payment $payment,
		?string $name = null,
		?float  $price = null
	): void
	{
		try {
			$orderPayment = $order->getPayment();

			if (!$orderPayment) {
				if (!$name) {
					$name = $payment->getName();
				}

				if (!$price) {
					$price = $payment->getPrice();
				}
			}

			$orderPayment->setPayment($payment);

			if ($price !== null) {
				$orderPayment->setPrice($price);
			}

			if ($name) {
				$orderPayment->setName($name);
			}

			$this->em->persist($orderPayment);
			$this->em->flush();
		} catch (\Exception $e) {
			Debugger::log($e, 'eshoporders');

			throw $e;
		}
	}
}
