<?php declare(strict_types = 1);

namespace EshopOrders\Model;

use Contributte\Translation\Translator;
use Core\Model\Event\EventDispatcher;
use Core\Model\Helpers\Arrays;
use Core\Model\Helpers\BaseEntityService;
use Core\Model\Mailing\TemplateFactory;
use Core\Model\Settings;
use Core\Model\Sites;
use EshopCatalog\FrontModule\Model\Sellers;
use EshopOrders\FrontModule\Model\Helpers\SchemaGenerator;
use EshopOrders\Model\Dao\DaoOrderStatusEmail;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderFlag;
use EshopOrders\Model\Entities\OrderStatus;
use EshopOrders\Model\Entities\Status;
use EshopOrders\Model\Event\EmailEvent;
use EshopOrders\Model\Event\OrderStatusEvent;
use EshopOrders\Model\Helpers\QrGenerator;
use EshopOrders\Model\Utils\Helpers;
use EshopOrders\Model\Utils\Strings;
use Nette\Mail\Mailer;
use Nette\Mail\Message;
use Tracy\Debugger;
use Users\Model\Entities\User;
use Users\Model\Http\UserStorage;

/**
 * @method Status getReference($id)
 * @method Status|null get($id)
 */
class Statuses extends BaseEntityService
{
	protected $entityClass = Status::class;

	protected Translator      $translator;
	protected Mailer          $mailer;
	protected TemplateFactory $mailTemplateFactory;
	protected UserStorage     $userStorage;
	protected EventDispatcher $eventDispatcher;
	protected Sellers         $sellers;
	protected Sites           $sites;
	protected Payments        $paymentsService;
	protected QrGenerator     $qrGenerator;
	protected Settings        $settings;
	protected SchemaGenerator $schemaGenerator;

	public function __construct(
		Translator      $translator,
		TemplateFactory $mailTemplateFactory,
		Mailer          $mailer,
		UserStorage     $userStorage,
		EventDispatcher $eventDispatcher,
		Sellers         $sellers,
		Sites           $sites,
		Payments        $payments,
		QrGenerator     $qrGenerator,
		Settings        $settings,
		SchemaGenerator $schemaGenerator
	)
	{
		$this->translator          = $translator;
		$this->mailer              = $mailer;
		$this->mailTemplateFactory = $mailTemplateFactory;
		$this->userStorage         = $userStorage;
		$this->eventDispatcher     = $eventDispatcher;
		$this->sellers             = $sellers;
		$this->sites               = $sites;
		$this->paymentsService     = $payments;
		$this->qrGenerator         = $qrGenerator;
		$this->settings            = $settings;
		$this->schemaGenerator     = $schemaGenerator;
	}

	public function setPosition(int $id, int $position): bool
	{
		if ($item = $this->get($id)) {
			$item->setPosition($position);
			$this->em->persist($item);
			$this->em->flush();

			return true;
		}

		return false;
	}

	/** @return Status[] */
	public function getAll(): array
	{
		return $this->getEr()->createQueryBuilder('s')
			->orderBy('s.position', 'asc')
			->getQuery()->getResult();
	}

	public function changeStatus(array $ids, string $status, bool $sendEmail = true): bool
	{
		try {
			$canProcess = [];
			foreach ($ids as $id) {
				$canProcess[$id] = $id;
			}

			foreach ($this->em->getRepository(OrderStatus::class)->createQueryBuilder('os')
				         ->select('IDENTITY(os.order) as order, IDENTITY(os.status) as stat')
				         ->where('os.order IN (:ids)')
				         ->andWhere('os.deleted IS NULL')
				         ->setParameter('ids', $ids)
				         ->orderBy('os.created', 'asc')
				         ->getQuery()->getArrayResult() as $row) {
				if ($row['stat'] === $status) {
					unset($canProcess[$row['order']]);
				}
			}

			foreach ($canProcess as $id) {
				/** @var Order|null $order */
				$order = $this->em->getRepository(Order::class)->find($id);
				if (!$order) {
					continue;
				}

				$texts         = new DaoOrderStatusEmail();
				$texts->status = $status;

				if ($status === OrderStatus::STATUS_IS_PAID) {
					$order->isPaid = 1;

					$texts->statusText = $this->translator->translate('eshopOrdersFront.emails.orderIsPaid');

					$this->em->persist($order);
					$this->em->flush();

					if ($sendEmail) {
						$this->sendOrderStatusEmail($order, null, $texts, $status);
					}
				} else {
					/** @var User|null $userRef */
					$userRef = $this->userStorage->getIdentity()
						? $this->em->getReference(User::class, $this->userStorage->getIdentity()->getId())
						: null;

					$this->em->getConnection()->executeStatement("INSERT INTO eshop_orders__order_status 
    					(order_id, status_id, created, created_by) VALUES (?, ?, ?, ?)", [
						$id, $status, (new \DateTime())->format('Y-m-d H:i:s'), $userRef ? $userRef->getId() : null,
					]);

					/** @var OrderStatus $entity */
					$entity = $this->em->getRepository(OrderStatus::class)->find($this->em->getConnection()->lastInsertId());

					$this->eventDispatcher->dispatch(new OrderStatusEvent($order, $entity->getStatus()->getId()), Order::class . '::changeStatus');

					if ($sendEmail) {
						$this->sendOrderStatusEmail($order, $entity, $texts, $status);
					}
				}
			}
		} catch (\Exception $e) {
			Debugger::log($e, 'status-change');

			return false;
		}

		return true;
	}

	/**
	 * @param Order                    $order
	 * @param OrderStatus|null         $orderStatus
	 * @param DaoOrderStatusEmail|null $texts
	 */
	public function sendOrderStatusEmail(Order $order, ?OrderStatus $orderStatus = null, ?DaoOrderStatusEmail $texts = null, ?string $statusId = null): bool
	{
		try {
			if (!$orderStatus && !$texts
				|| !EshopOrdersConfig::load('orderForm.send', false)
				|| !$order->getAddressInvoice()
				|| !$order->getAddressInvoice()->getEmail()
				|| $orderStatus && $orderStatus->getStatus() && !$orderStatus->getStatus()->sendToCustomer) {
				return false;
			}

			if (in_array($order->site->getIdent(), EshopOrdersConfig::load('orderForm.sendDisableSites') ?: [], true)) {
				return false;
			}

			if (!$statusId && $orderStatus) {
				$statusId = $orderStatus->getStatus()->getId();
			}

			if (in_array($statusId, EshopOrdersConfig::load('orderStatus.excludeStatusEmail', []))) {
				return false;
			}

			// Vypnutí odeslání emailu po odeslání faktury pro určité stavyOrder
			if (EshopOrdersConfig::load('invoice.invoiceInEmailIsEndStatusEmail') && $statusId
				&& in_array($statusId, [OrderStatus::STATUS_IS_PAID, OrderStatus::STATUS_FINISHED])
				&& $order->hasFlag(OrderFlag::TYPE_INVOICE_SENT)) {
				return false;
			}

			// Vypnutí odeslání emailu se stavem zaplaceno u dobírek
			if ($statusId === OrderStatus::STATUS_IS_PAID
				&& in_array((string) $order->getSpeditionIdent(), EshopOrdersConfig::load('disableIsPaidEmailForSpedition', []))) {
				return false;
			}

			if ($orderStatus && !$texts) {
				$texts          = new DaoOrderStatusEmail();
				$texts->status  = $orderStatus->getStatus()->getId();
				$texts->message = $orderStatus->getMessage();
			}

			$this->translator->setLocale($order->lang);
			Sites::$currentIdentOverride = $order->site->getIdent();
			Sites::$currentLangOverride  = $order->lang;

			$file       = TEMPLATES_DIR . '/Front/default/EshopOrders/_emails/status.latte';
			$originFile = __DIR__ . '/../FrontModule/_emails/status.latte';
			$template   = $this->mailTemplateFactory->create($order->site->getIdent(), $order->lang);
			$seller     = $this->sellers->getSellerForSite($order->site->getIdent());
			$site       = $this->sites->getSites(false)[$order->site->getIdent()] ?? null;

			if (!file_exists($file) || !$site) {
				$file = $originFile;
			}

			$subject = $this->translator->translate('eshopOrders.emails.subjectStatus',
				['orderId' => $order->getId(), 'siteName' => $template->siteName]);

			$country = $order->getAddressInvoice() && $order->getAddressInvoice()->getCountry()
				? Strings::upper($order->getAddressInvoice()->getCountry()->getId())
				: null;

			$customer      = $order->getCustomer();
			$canShowQrCode = EshopOrdersConfig::load('emails.showQrPayment')
				&& $country && Arrays::contains(EshopOrdersConfig::load('showQrForCountries'), $country)
				&& $seller
				&& ($order->getPaymentIdent() === 'transfer' || EshopOrdersConfig::load('emails.showQrPaymentInAllEmails'))
				&& (
					!$customer
					|| !$customer->getGroupCustomers()
					|| !Arrays::contains(EshopOrdersConfig::load('emails.hideQrPaymentForCustomerGroup'), $customer->getGroupCustomers()->getId())
				);

			$orderSchema = $this->schemaGenerator->generateOrderSchema($order, $site->getDomains()[$order->lang] ?? null, $seller);
			if ($orderSchema) {
				$orderSchema = $this->schemaGenerator->getSchemaForEmail($orderSchema);
			}

			$template->setFile($file);
			$template->setParameters([
				'templateParts'  => __DIR__ . '/../FrontModule/_emails/_parts.latte',
				'subject'        => $subject,
				'order'          => $order,
				'orderStatus'    => $orderStatus,
				'texts'          => $texts,
				'orderId'        => $order->getId(),
				'originTemplate' => $originFile,
				'canPayWithCard' => EshopOrdersConfig::load('sendPayLinkInEmail')
					&& in_array($order->getPaymentIdent(), ['card', 'essox'])
					&& $this->paymentsService->getByIdent('card', true),
				'qrCode'         => $canShowQrCode ? $this->qrGenerator->getQrCodeByOrderAndSeller($order, $seller) : null,
				'schemaOrder'    => $orderSchema,
			]);

			$siteIdent   = $order->site->getIdent();
			$sellerEmail = Helpers::getSellerEmail($seller, $order->site, $order->lang);
			$sellerName  = Helpers::getSellerName($seller, $siteIdent, $order->lang);

			$message = new Message();
			$message->setFrom($sellerEmail, $sellerName);

			if (Arrays::contains(EshopOrdersConfig::load('orderForm.termsAndConditionsAttachmentInStatus'), $texts->status)) {
				$termsAndConditionsPdf = $this->settings->get('eshopOrdersTermsAndConditionsPdf')[$this->translator->getLocale()] ?? null;
				if ($termsAndConditionsPdf && file_exists(WWW_DIR . $termsAndConditionsPdf)) {
					$message->addAttachment(
						$this->translator->translate('eshopOrdersFront.cart.termsAndConditions') . '.pdf',
						(string) file_get_contents(WWW_DIR . $termsAndConditionsPdf),
					);
				}
			}

			if ($seller->sendStatusToSeller) {
				$message->addTo($sellerEmail, $sellerName);
			}

			$bcc = EshopOrdersConfig::load('orderForm.bcc');
			if ($bcc) {
				foreach (explode(',', $bcc) as $v) {
					$message->addBcc(trim($v));
				}
			}

			$replyTo = EshopOrdersConfig::load('orderForm.replyTo');
			if ($replyTo) {
				$message->addReplyTo($replyTo);
			}

			if (
				EshopOrdersConfig::load('orderForm.sendTo') === 'customer'
				&& $order->getCustomer()
			) {
				$user    = $order->getCustomer()->getUser();
				$toEmail = $user->getEmail();
				$toName  = $user->getName() . ' ' . $user->getLastname();
			} else {
				$toEmail = $order->getAddressInvoice()->getEmail();
				$toName  = $order->getAddressInvoice()->getFirstName() . ' ' . $order->getAddressInvoice()->getLastName();
			}

			$message->addTo($toEmail, $toName);

			if (
				Arrays::contains(EshopOrdersConfig::load('order.sendEmailStatusToInvoiceAddress') ?? [], $statusId)
				&& $order->getAddressInvoiceRaw()
				&& $order->getAddressInvoiceRaw()->getEmail()
				&& $order->getAddressInvoiceRaw()->getEmail() !== $toEmail
			) {
				$message->addCc(
					$order->getAddressInvoiceRaw()->getEmail(),
					$order->getAddressInvoiceRaw()->getFirstName() . ' ' . $order->getAddressInvoiceRaw()->getLastName()
				);
			}

			$statusString = $orderStatus ? $orderStatus->getStatus()->getId() : $texts->status;
			if ($statusString) {
				$this->eventDispatcher->dispatch(new EmailEvent($template, $message, $order, $statusString, $orderStatus), 'eshopOrders.emailChangeOrderStatus');
			}

			$message->setHtmlBody($template->renderToString());
			$this->mailer->send($message);
		} catch (\Exception $e) {
			Debugger::log('Order email failed - ' . $order->getId(), 'eshopOrderEmail');
			Debugger::log($e, 'eshopOrderEmail');
		}

		return true;
	}
}
