<?php declare(strict_types = 1);

namespace EshopOrders\Model\Subscribers;

use Contributte\Translation\Translator;
use Core\Model\Entities\EntityManagerDecorator;
use EshopOrders\FrontModule\Model\Affiliate;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderFlag;
use EshopOrders\Model\Entities\OrderStatus;
use EshopOrders\Model\EshopOrdersConfig;
use EshopOrders\Model\Event\EmailEvent;
use EshopOrders\Model\Event\OrderStatusEvent;
use EshopOrders\Model\Invoices;
use EshopOrders\Model\Orders;
use EshopOrders\Model\Pdf\IInvoicePdfBuilderFactory;
use EshopOrders\Model\Statuses;
use Mpdf\Output\Destination;
use Nette\Utils\Html;
use Nette\Utils\Validators;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class OrderSubscriber implements EventSubscriberInterface
{
	public static bool $enableSendInvoice = false;

	protected Translator                $translator;
	protected EntityManagerDecorator    $em;
	protected Statuses                  $statusesService;
	protected Invoices                  $invoices;
	protected IInvoicePdfBuilderFactory $invoicePdfBuilder;
	protected Orders                    $orders;
	protected Affiliate                 $affiliate;

	public function __construct(
		Translator                $translator,
		EntityManagerDecorator    $em,
		Statuses                  $statuses,
		Invoices                  $invoices,
		IInvoicePdfBuilderFactory $invoicePdfBuilder,
		Orders                    $orders,
		Affiliate                 $affiliate
	)
	{
		$this->translator        = $translator;
		$this->em                = $em;
		$this->statusesService   = $statuses;
		$this->invoices          = $invoices;
		$this->invoicePdfBuilder = $invoicePdfBuilder;
		$this->orders            = $orders;
		$this->affiliate         = $affiliate;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			Order::class . '::changeStatus'      => 'orderChangeStatus',
			'eshopOrders.emailChangeOrderStatus' => 'emailChangeOrderStatus',
		];
	}

	public function orderChangeStatus(OrderStatusEvent $event): void
	{
		if ($event->status === OrderStatus::STATUS_IS_PAID) {
			$this->statusesService->changeStatus([$event->order->getId()], OrderStatus::STATUS_IS_PAID);
		} else if ($event->status === OrderStatus::STATUS_CANCELED) {
			if (EshopOrdersConfig::load('affiliate.enable')) {
				$this->affiliate->cancelOrder($event->order);
			}
		}
	}

	public function emailChangeOrderStatus(EmailEvent $event): void
	{
		$order  = $event->order;
		$status = $event->status;

		if (EshopOrdersConfig::load('invoice.enable')
			&& in_array($status, EshopOrdersConfig::load('invoice.orderStatusSendTrigger', []))
			&& !$order->hasFlag(OrderFlag::TYPE_INVOICE_SENT)) {

			if (EshopOrdersConfig::load('invoice.allowUploadCustomInvoiceFile', false)
				&& (!self::$enableSendInvoice || !$order->enableInvoiceGeneration)
				&& !$order->getInvoice()
			) {
				return;
			}

			$invoice = $this->invoices->getOneByOrder($this->orders->get((int) $event->order->getId()));
			if ($invoice === null) {
				$invoice = $this->invoices->getOneByOrder($this->orders->get((int) $event->order->getId()), true);
			}
			$pdfBuilder = $this->invoicePdfBuilder->create($invoice);
			$event->message->addAttachment($pdfBuilder->getFileName(), $pdfBuilder->render(Destination::STRING_RETURN));

			$event->addCustomHtml(Html::el('p')->setText($this->translator->translate('eshopOrdersFront.emails.invoiceIsInAttachment')));

			if (EshopOrdersConfig::load('customerGroup.allowInvoiceCopyToEmail')) {
				$customerGroup = $order->getCustomer() ? $order->getCustomer()->getGroupCustomers() : null;

				if ($customerGroup && $customerGroup->invoiceCopyEmail) {
					foreach (explode(',', (string) $customerGroup->invoiceCopyEmail) as $email) {
						$email = trim($email);

						if (Validators::isEmail($email)) {
							$event->message->addBcc($email);
						}
					}
				}
			}

			$this->em->getConnection()->insert('eshop_orders__order_flag', [
				'type'     => OrderFlag::TYPE_INVOICE_SENT,
				'state'    => 1,
				'order_id' => $order->getId(),
			]);
		}
	}
}
