<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Components\Order;

use Core\AdminModule\Model\Sites;
use Core\Model\Entities\Site;
use Core\Model\UI\BaseControl;
use Currency\DI\CurrencyExtension;
use EshopOrders\AdminModule\Model\Payments;
use EshopOrders\AdminModule\Model\Speditions;
use EshopOrders\AdminModule\Model\Statuses;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderItem;
use EshopOrders\Model\Entities\OrderStatus;
use EshopOrders\Model\Orders;
use Core\Model\Entities\QueryBuilder;
use Nette\Utils\DateTime;
use Nette\Utils\Html;
use Core\Model\Templating\Filters\Price as PriceFilter;

class OrdersGrid extends BaseControl
{
	/** @var Orders */
	protected $ordersService;

	/** @var array */
	protected $config;

	/** @var array */
	protected $exportCallbacks;

	/** @var Speditions */
	protected $speditionsService;

	/** @var Payments */
	protected $paymentsService;

	/** @var Statuses */
	protected $statusesService;

	/** @var Sites */
	protected $sitesService;

	/** @var PriceFilter */
	protected $priceFilter;

	public function __construct($data, Orders $orders, Speditions $speditions, Payments $payments, Statuses $statuses,
	                            Sites $sites, PriceFilter $priceFilter)
	{
		$this->ordersService     = $orders;
		$this->config            = $data;
		$this->speditionsService = $speditions;
		$this->paymentsService   = $payments;
		$this->statusesService   = $statuses;
		$this->sitesService      = $sites;
		$this->priceFilter       = $priceFilter;
	}

	public function render()
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentGrid()
	{
		$grid = $this->createGrid();
		$grid->setStrictSessionFilterValues(false);

		$qb = $this->ordersService->getEr()->createQueryBuilder('o')
			->addSelect('ai, ad, payment, spedition, discounts, site, currency')
			->leftJoin('o.addressInvoice', 'ai')
			->leftJoin('o.addressDelivery', 'ad')
			->innerJoin('o.payment', 'payment')
			->innerJoin('o.spedition', 'spedition')
			->leftJoin('o.orderDiscounts', 'discounts')
			->leftJoin('o.currency', 'currency')
			->innerJoin('o.site', 'site')
			->orderBy('o.id', 'DESC')
			->groupBy('o.id');

		/** @var OrderStatus[][] $orderStatuses */
		$orderStatuses = [];
		/** @var Site $sites */
		$sites = [];
		foreach ($this->sitesService->getAll() as $row)
			$sites[$row->getIdent()] = $row->getIdent();

		$grid->setDataSource($qb);

		$grid->getDataSource()->onDataLoaded[] = function($items) use (&$orderStatuses, &$orderItems) {
			$ids = array_map(function($v) { return $v->getId(); }, $items);

			foreach ($this->em->getRepository(OrderStatus::class)->createQueryBuilder('os')
				         ->addSelect('s')
				         ->innerJoin('os.status', 's')
				         ->where('os.order IN (:orders)')->setParameter('orders', $ids)
				         ->orderBy('os.created', 'DESC')
				         ->getQuery()->getResult() as $row) {
				/** @var OrderStatus $row */
				if ($row->isDeleted())
					continue;

				$orderStatuses[$row->getOrder()->getId()][] = $row;
			}

			foreach ($this->em->getRepository(OrderItem::class)->createQueryBuilder('oi')
				         ->where('oi.order IN (:orders)')->setParameter('orders', $ids)
				         ->getQuery()->getResult() as $row)
				$orderItems[$row->getOrder()->getId()][] = $row;
		};

		//Columns
		$grid->addColumnLink('id', 'eshopOrders.default.orderNumberShort', 'Default:editOrder')->setAlign('right');
		$grid->addColumnText('firstName', 'default.name', 'addressDelivery.firstName');
		$grid->addColumnText('lastName', 'eshopOrders.default.lastName', 'addressDelivery.lastName');
		$grid->addColumnText('email', 'eshopOrders.default.email')->setRenderer(function(Order $row) {
			return Html::el('a', ['href' => 'mailto:' . $row->getAddressDelivery()->getEmail(), 'target' => '_blank'])
				->setText($row->getAddressDelivery()->getEmail());
		});
		$grid->addColumnPhone('phone', 'eshopOrders.default.phone', 'addressDelivery.phone', 'addressDelivery.country.id');
		$grid->addColumnText('spedition', 'eshopOrders.default.spedition', 'spedition.name');
		$grid->addColumnText('payment', 'eshopOrders.default.payment', 'payment.name');
		$grid->addColumnDateTime('created', 'eshopOrders.defaultGrid.createdTime', 'createdTime')->setRenderer(function(Order $row) use (&$orderStatuses) {
			$statuses = $orderStatuses[$row->getId()];

			return $statuses ? end($statuses)->getCreated()->format('d. m. Y H:i') : '';
		})->getElementPrototype('th')->addClass('w1nw');
		$grid->addColumnText('status', 'eshopOrders.default.status')->setRenderer(function(Order $row) use (&$orderStatuses) {
			$statuses = $orderStatuses[$row->getId()];

			return $statuses ? $statuses[0]->getStatus()->getName() : '';
		});
		$grid->addColumnNumber('price', 'eshopOrders.default.price')->setRenderer(function(Order $row) use (&$orderItems) {
			if ($orderItems[$row->getId()])
				$row->setOrderItems($orderItems[$row->getId()]);

			if ($row->currency) {
				$price2    = $row->getPrice(true);
				$code2     = $row->currency->code;
				$decimals2 = $row->currency->decimals;
			}

			return $this->priceFilter
				->renderTwoPrices($row->getPrice(), $row->getDefaultCurrency(), $price2 ?? null, $code2 ?? null, $decimals2 ?? null);
		})->getElementPrototype('th')->addClass('w1nw');

		if (class_exists(CurrencyExtension::class)) {
			$grid->addColumnText('currency', 'eshopOrders.default.currency', 'currencyCode');
		}

		if (count($sites) > 1)
			$grid->addColumnText('site', 'eshopOrders.default.eshop', 'site.ident');

		// Filter
		$grid->addFilterText('id', '');
		$grid->addFilterText('firstName', '', ['ad.firstName', 'ai.firstName']);
		$grid->addFilterText('lastName', '', ['ad.lastName', 'ai.lastName']);
		$grid->addFilterText('email', '', ['ad.email', 'ai.email']);
		$grid->addFilterText('phone', '', ['ad.phone', 'ai.phone']);
		$grid->addFilterSelect('spedition', '', $this->speditionsService->getForSelectOption(), 'spedition.spedition');
		$grid->addFilterSelect('payment', '', $this->paymentsService->getForSelectOption(), 'payment.payment');
		$grid->addFilterSelect('status', '', $this->statusesService->getForSelectOption())->setCondition(function(QueryBuilder $qb, $value) {
			$join = $this->em->getRepository(OrderStatus::class)->createQueryBuilder('os2')
				->select('GROUP_CONCAT(IDENTITY(os2.status))')
				->andWhere('os2.order = o.id')
				->andWhere('os2.deleted IS NULL OR os2.deleted = 0')
				->groupBy('os2.order')
				->orderBy('os2.order', 'DESC')
				->getQuery();

			$qb->addSelect("(" . $join->getDQL() . ") as hidden statusFilter")
				->andHaving('statusFilter LIKE :status2')
				->setParameter('status2', "%$value");
		});
		$grid->addFilterDateRange('created', '')->setCondition(function(QueryBuilder $qb, $values) {
			if ($values->from || $values->to) {
				if (!in_array('oStatuses', $qb->getAllAliases()))
					$qb->innerJoin('o.orderStatuses', 'oStatuses');
				$qb->andWhere('oStatuses.status = :statusName2')->setParameter('statusName2', 'created');
			}

			if ($values->from)
				$qb->andWhere('oStatuses.created >= :statusFrom')
					->setParameter('statusFrom', DateTime::createFromFormat('j. n. Y', $values->from)->setTime(0, 0, 0));
			if ($values->to)
				$qb->andWhere('oStatuses.created <= :statusTo')
					->setParameter('statusTo', DateTime::createFromFormat('j. n. Y', $values->to)->setTime(23, 59, 59));
		});
		$grid->addFilterRange('price', '')->setCondition(function(QueryBuilder $qb, $values) {
			$qb->innerJoin('o.orderItems', 'items')
				->addSelect('SUM(items.price) + payment.price + spedition.price + IFNULL(SUM(discounts.price), 0) as hidden hPrice');

			if ($values->from)
				$qb->andHaving('hPrice >= :priceFrom')->setParameter('priceFrom', (float) $values->from);
			if ($values->to)
				$qb->andHaving('hPrice <= :priceTo')->setParameter('priceTo', (float) $values->to);
		});

		if (count($sites) > 1)
			$grid->getColumn('site')->setFilterSelect([null => ''] + $sites);

		// ItemDetail
		$grid->setItemsDetail($this->getTemplateFile('OrdersGridDetail', 'Order'));

		// Actions
		$grid->addAction('edit', '', 'Default:editOrder')->setIcon('edit')->setBsType('primary');
		if ($this->config['allowOrderDelete']) {
			$grid->addAction('delete', '', 'delete!')->setConfirm('eshopOrders.orderForm.reallyDelete')->setIcon('times')->setBsType('danger');
		}

		// Group action
		$grid->addGroupSelectAction('eshopOrders.defaultGrid.changeStatus', $this->statusesService->getForSelectOption())->onSelect[] = [$this,
			'gridChangeStatus'];
		if ($this->exportCallbacks) {
			foreach ($this->exportCallbacks as $v) {
				$grid->addGroupAction($v['title'])->onSelect[] = $v['callback'];
			}
		}

		$grid->addAction('pdf', '')->setRenderer(function(Order $order) {
			if ($order->getInvoice() === null) {
				return '';
			}

			return Html::el('a', ['href'  => $this->presenter->link('pdf!', $order->getInvoice()->getId()),
			                      'class' => 'btn btn-danger btn-xs', 'target' => '_blank'])
				->addHtml(Html::el('i', ['class' => 'fas fa-file-pdf']));
		});

		// Columns prototype
		$grid->getColumn('id')->getElementPrototype('th')->class[]      = 'w1nw';
		$grid->getColumn('created')->getElementPrototype('td')->class[] = 'w1nw';

		return $grid;
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */

	public function handleDelete($id)
	{
		$presenter = $this->getPresenter();
		if ($this->config['allowOrderDelete']) {
			if ($this->ordersService->remove($id))
				$presenter->flashMessageSuccess('default.removed');
			else
				$presenter->flashMessageDanger('default.removeFailed');
		} else {
			$presenter->flashMessageDanger('eshopOrders.orderForm.removeDenied');
		}

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	/*******************************************************************************************************************
	 * =================  Grid function
	 */

	public function gridChangeStatus(array $ids, string $value): void
	{
		if (!$value || empty($ids))
			return;

		$presenter = $this->getPresenter();
		if ($this->statusesService->changeStatus($ids, $value))
			$presenter->flashMessageSuccess('eshopOrders.defaultGrid.publishChanged');
		else
			$presenter->flashMessageDanger('eshopOrders.defaultGrid.publishChangeFailed');

		if ($presenter->isAjax()) {
			$presenter->redrawControl('flashes');
			$this['grid']->reload();
		} else {
			$this->redirect('this');;
		}
	}

	/*******************************************************************************************************************
	 * =================  GET SET
	 */

	public function addExportCallback(string $key, string $title, callable $callback): self
	{
		$this->exportCallbacks[$key] = [
			'title'    => $title,
			'callback' => $callback,
		];

		return $this;
	}
}
