<?php declare(strict_types = 1);

namespace EshopOrders\AdminModule\Components\PaymentSpedition;

use Core\AdminModule\Model\Sites;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Nette\InvalidArgumentException;
use Nette\Utils\ArrayHash;
use EshopOrders\Model\Entities\PaymentSpedition;
use EshopOrders\AdminModule\Model\Speditions;
use EshopOrders\Model\Payments;
use EshopOrders\Model\PaymentSpeditions;

class PaymentSpeditionForm extends BaseControl
{
	/** @var int @persistent */
	public $id;

	/** @var PaymentSpeditions */
	protected $paymentSpeditionsService;

	/** @var PaymentSpedition */
	public $paymentSpedition;

	/** @var Speditions */
	protected $speditionsService;

	/** @var Payments */
	protected $paymentsService;

	/** @var Sites */
	protected $sitesService;

	public function __construct(PaymentSpeditions $paymentSpeditions, Speditions $speditions, Payments $payments, Sites $sites)
	{
		$this->paymentSpeditionsService = $paymentSpeditions;
		$this->speditionsService        = $speditions;
		$this->paymentsService          = $payments;
		$this->sitesService             = $sites;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax();

		$paymentChoices = $speditionChoices = [];
		$speditions     = $this->speditionsService->getAll();
		$payments       = $this->paymentsService->getAll();
		foreach ($speditions as $s) {
			$speditionChoices[$s->getId()] = $s->getName() . ' - ' . $s->getPrice() . ' Kč';
		}
		foreach ($payments as $p) {
			$paymentChoices[$p->getId()] = $p->getName() . ' - ' . $p->getPrice() . ' Kč';
		}
		$form->addSelect('spedition', 'eshopOrders.orderPage.spedition', $speditionChoices)->setRequired();
		$form->addSelect('payment', 'eshopOrders.orderPage.payment', $paymentChoices)->setRequired();
		$form->addSelect('site', 'eshopOrders.paymentSpedition.site', $this->sitesService->getOptionsForSelect())
			->setRequired();

		$form->addBool('isPublished', 'default.isActive')->setDefaultValue(1);
		$form->addSaveCancelControl('saveControl');

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		try {
			$site = $this->sitesService->getReference($values->site);

			if ($this->id) {
				$paymentSpedition = $this->paymentSpeditionsService->get($this->id);
				$flashMessage     = 'eshopOrders.paymentSpeditionForm.edited';
			} else {
				$paymentSpedition = new PaymentSpedition($site);
				$flashMessage     = 'eshopOrders.paymentSpeditionForm.added';
			}

			$paymentSpedition->setPayment($this->paymentsService->get($values->payment));
			$paymentSpedition->setSpedition($this->speditionsService->get($values->spedition));
			$paymentSpedition->isPublished = $values->isPublished;
			$paymentSpedition->site        = $site;

			$this->em->persist($paymentSpedition)->flush();
			$form->addCustomData('paymentSpeditionId', $paymentSpedition->getId());
			$this->getPresenter()->flashMessageSuccess($flashMessage);

			return true;
		} catch (\Exception $e) {
			$form->addError($e->getMessage());
		}

		return false;
	}

	public function setPaymentSpedition($id): void
	{
		$this->id               = $id;
		$this->paymentSpedition = $this->paymentSpeditionsService->get($id);

		if (!$this->paymentSpedition)
			throw new InvalidArgumentException();

		$m = $this->paymentSpedition;
		$f = $this['form'];
		$d = [
			'payment'     => $m->getPayment()->getId(),
			'spedition'   => $m->getSpedition()->getId(),
			'isPublished' => $m->isPublished,
		];

		if (array_key_exists($m->site, $f['site']->getItems()))
			$d['site'] = $m->site->getIdent();

		$f->setDefaults($d);
	}
}
